<?php
// custom_fields_helper.php

/**
 * Calculate custom field pay based on calculation type
 */
function calculateCustomFieldPay($field_data, $value, $gross_pay, $employee_data, $custom_rates = []) {
    $calculated_pay = 0;
    
    switch($field_data['calculation_type']) {
        case 'multiply_rate':
            // Get rate from employee data or custom rates
            $rate_field = $field_data['rate_field'];
            $rate = 0;
            
            // Check standard rates first
            switch($rate_field) {
                case 'hourly_rate':
                    $rate = $employee_data['dAmount'] ?? 16.00;
                    break;
                case 'solo_bonus_rate':
                    $rate = $employee_data['solo_rate'] ?? 3.00;
                    break;
                case 'mileage_rate':
                    $rate = $employee_data['miles_rate'] ?? 0.25;
                    break;
                case 'miscellaneous_rate':
                    $rate = $employee_data['miscellaneous_rate'] ?? 1.00;
                    break;
                default:
                    // Check custom rates
                    $rate = $custom_rates[$rate_field] ?? 0;
            }
            
            $calculated_pay = $value * $rate;
            break;
            
        case 'fixed_amount':
            $calculated_pay = $value;
            break;
            
        case 'percentage_of_gross':
            $calculated_pay = ($value / 100) * $gross_pay;
            break;
            
        default:
            $calculated_pay = $value;
    }
    
    return round($calculated_pay, 2);
}

/**
 * Get employee custom rates
 */
function getEmployeeCustomRates($employee_id) {
    global $con;
    
    $query = "SELECT custom_fields_rates FROM usermst WHERE id = $employee_id";
    $result = mysqli_query($con, $query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $data = mysqli_fetch_assoc($result);
        return json_decode($data['custom_fields_rates'] ?? '{}', true) ?: [];
    }
    
    return [];
}

/**
 * Update employee custom rate
 */
function updateEmployeeCustomRate($employee_id, $field_slug, $rate) {
    global $con;
    
    $current_rates = getEmployeeCustomRates($employee_id);
    $current_rates[$field_slug] = floatval($rate);
    
    $rates_json = json_encode($current_rates);
    mysqli_query($con, "UPDATE usermst SET custom_fields_rates = '$rates_json' WHERE id = $employee_id");
    
    return true;
}

/**
 * Get all active custom fields with employee-specific rates
 */
function getCustomFieldsWithRates($employee_id = null) {
    global $con;
    
    $query = "SELECT * FROM custom_payroll_fields WHERE is_active = 1 ORDER BY display_order";
    $result = mysqli_query($con, $query);
    
    $fields = [];
    $custom_rates = $employee_id ? getEmployeeCustomRates($employee_id) : [];
    
    while ($field = mysqli_fetch_assoc($result)) {
        $field['employee_rate'] = $custom_rates[$field['field_slug']] ?? $field['default_value'];
        $fields[] = $field;
    }
    
    return $fields;
}

/**
 * Get custom field values for a weekly report
 */
function getReportCustomFields($report_id) {
    global $con;
    
    $query = "
        SELECT cpf.*, wrcf.field_value, wrcf.text_value, wrcf.calculated_pay
        FROM weekly_report_custom_fields wrcf
        JOIN custom_payroll_fields cpf ON cpf.id = wrcf.custom_field_id
        WHERE wrcf.weekly_report_id = $report_id
        ORDER BY cpf.display_order
    ";
    
    $result = mysqli_query($con, $query);
    $fields = [];
    
    while ($field = mysqli_fetch_assoc($result)) {
        $fields[] = $field;
    }
    
    return $fields;
}

/**
 * Ensure custom fields tables exist
 */
function ensureCustomFieldsTables() {
    global $con;
    
    // Check if custom_payroll_fields table exists
    $check = mysqli_query($con, "SHOW TABLES LIKE 'custom_payroll_fields'");
    if (!$check || mysqli_num_rows($check) == 0) {
        // Create table
        $sql = "CREATE TABLE IF NOT EXISTS custom_payroll_fields (
            id INT AUTO_INCREMENT PRIMARY KEY,
            field_name VARCHAR(100) NOT NULL,
            field_slug VARCHAR(100) NOT NULL UNIQUE,
            field_type ENUM('hours', 'dollars', 'miles', 'units', 'percentage', 'text', 'number') NOT NULL,
            unit_label VARCHAR(50),
            default_value DECIMAL(10,2) DEFAULT 0.00,
            is_active BOOLEAN DEFAULT 1,
            is_required BOOLEAN DEFAULT 0,
            display_order INT DEFAULT 0,
            calculation_type ENUM('multiply_rate', 'fixed_amount', 'percentage_of_gross') DEFAULT 'multiply_rate',
            rate_field VARCHAR(50) DEFAULT 'miscellaneous_rate',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )";
        
        if (!mysqli_query($con, $sql)) {
            error_log("Failed to create custom_payroll_fields table: " . mysqli_error($con));
        }
    }
    
    // Check if weekly_report_custom_fields table exists
    $check = mysqli_query($con, "SHOW TABLES LIKE 'weekly_report_custom_fields'");
    if (!$check || mysqli_num_rows($check) == 0) {
        $sql = "CREATE TABLE IF NOT EXISTS weekly_report_custom_fields (
            id INT AUTO_INCREMENT PRIMARY KEY,
            weekly_report_id INT NOT NULL,
            custom_field_id INT NOT NULL,
            field_value DECIMAL(10,2),
            text_value TEXT,
            calculated_pay DECIMAL(10,2) DEFAULT 0.00,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (weekly_report_id) REFERENCES weekly_reports(id) ON DELETE CASCADE,
            FOREIGN KEY (custom_field_id) REFERENCES custom_payroll_fields(id) ON DELETE CASCADE,
            UNIQUE KEY unique_report_field (weekly_report_id, custom_field_id)
        )";
        
        if (!mysqli_query($con, $sql)) {
            error_log("Failed to create weekly_report_custom_fields table: " . mysqli_error($con));
        }
    }
}