<?php
session_start();
if (!isset($_SESSION['SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';

$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
unset($_SESSION['success'], $_SESSION['error']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_field'])) {
        $field_name = mysqli_real_escape_string($con, $_POST['field_name']);
        $field_slug = strtolower(preg_replace('/[^a-zA-Z0-9_]/', '_', $field_name));
        $field_type = $_POST['field_type'];
        $unit_label = mysqli_real_escape_string($con, $_POST['unit_label'] ?? '');
        $default_value = floatval($_POST['default_value'] ?? 0);
        $is_required = isset($_POST['is_required']) ? 1 : 0;
        
        $stmt = mysqli_prepare($con, "
            INSERT INTO custom_payroll_fields 
            (field_name, field_slug, field_type, unit_label, default_value, is_required, display_order) 
            VALUES (?, ?, ?, ?, ?, ?, 
                (SELECT COALESCE(MAX(display_order), 0) + 1 FROM custom_payroll_fields c)
            )
        ");
        mysqli_stmt_bind_param($stmt, "ssssdi", $field_name, $field_slug, $field_type, $unit_label, $default_value, $is_required);
        
        if (mysqli_stmt_execute($stmt)) {
            $_SESSION['success'] = "Custom field added successfully!";
        } else {
            $_SESSION['error'] = "Failed to add field: " . mysqli_error($con);
        }
        mysqli_stmt_close($stmt);
    }
    
    if (isset($_POST['update_field'])) {
        $field_id = intval($_POST['field_id']);
        $field_name = mysqli_real_escape_string($con, $_POST['field_name']);
        $field_type = $_POST['field_type'];
        $unit_label = mysqli_real_escape_string($con, $_POST['unit_label'] ?? '');
        $default_value = floatval($_POST['default_value'] ?? 0);
        $is_required = isset($_POST['is_required']) ? 1 : 0;
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order'] ?? 0);
        
        $stmt = mysqli_prepare($con, "
            UPDATE custom_payroll_fields 
            SET field_name = ?, field_type = ?, unit_label = ?, default_value = ?, 
                is_required = ?, is_active = ?, display_order = ?
            WHERE id = ?
        ");
        mysqli_stmt_bind_param($stmt, "sssdiiii", $field_name, $field_type, $unit_label, 
            $default_value, $is_required, $is_active, $display_order, $field_id);
        
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        $_SESSION['success'] = "Field updated successfully!";
    }
    
    if (isset($_POST['delete_field'])) {
        $field_id = intval($_POST['field_id']);
        
        // Check if field has data
        $check = mysqli_query($con, "SELECT COUNT(*) as count FROM weekly_report_custom_fields WHERE custom_field_id = $field_id");
        $data = mysqli_fetch_assoc($check);
        
        if ($data['count'] == 0) {
            mysqli_query($con, "DELETE FROM custom_payroll_fields WHERE id = $field_id");
            $_SESSION['success'] = "Field deleted successfully!";
        } else {
            $_SESSION['error'] = "Cannot delete field with existing data. Deactivate instead.";
        }
    }
    
    header('Location: custom_payroll_fields.php');
    exit;
}

// Fetch all custom fields
$fields = mysqli_query($con, "
    SELECT * FROM custom_payroll_fields 
    ORDER BY display_order, field_name
");
?>

<!DOCTYPE html>
<html>
<head>
    <title>Custom Payroll Fields</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <style>
        .field-card { margin-bottom: 15px; }
        .field-type-badge { font-size: 0.7rem; }
        .active-field { border-left: 4px solid #28a745; }
        .inactive-field { border-left: 4px solid #dc3545; opacity: 0.7; }
        .unit-label { color: #6c757d; font-size: 0.9rem; }
        .sortable-list { list-style: none; padding: 0; }
        .sortable-item { 
            padding: 10px; 
            margin-bottom: 5px; 
            background: #f8f9fa; 
            border: 1px solid #dee2e6; 
            cursor: move;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h3>Custom Payroll Fields</h3>
        
        <?php if($success_message): ?>
            <div class="alert alert-success"><?= $success_message ?></div>
        <?php endif; ?>
        
        <?php if($error_message): ?>
            <div class="alert alert-danger"><?= $error_message ?></div>
        <?php endif; ?>
        
        <!-- Add New Field Form -->
        <div class="card field-card">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0">Add New Payroll Field</h5>
            </div>
            <div class="card-body">
                <form method="post" class="row g-3">
                    <div class="col-md-3">
                        <label>Field Name</label>
                        <input type="text" name="field_name" class="form-control" required>
                    </div>
                    <div class="col-md-2">
                        <label>Field Type</label>
                        <select name="field_type" class="form-control" required>
                            <option value="hours">Hours</option>
                            <option value="dollars">Dollars</option>
                            <option value="miles">Miles</option>
                            <option value="units">Units</option>
                            <option value="percentage">Percentage</option>
                            <option value="number">Number</option>
                            <option value="text">Text</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label>Unit Label</label>
                        <input type="text" name="unit_label" class="form-control" 
                               placeholder="e.g., hours, miles, $">
                    </div>
                    <div class="col-md-2">
                        <label>Default Value</label>
                        <input type="number" step="0.01" name="default_value" 
                               class="form-control" value="0">
                    </div>
                    <div class="col-md-2 d-flex align-items-end">
                        <div class="form-check">
                            <input type="checkbox" name="is_required" class="form-check-input">
                            <label class="form-check-label">Required</label>
                        </div>
                    </div>
                    <div class="col-md-1 d-flex align-items-end">
                        <button name="add_field" class="btn btn-success">Add</button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Fields List -->
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="mb-0">Existing Fields (Drag to reorder)</h5>
            </div>
            <div class="card-body">
                <ul class="sortable-list" id="sortableFields">
                    <?php while($field = mysqli_fetch_assoc($fields)): ?>
                        <li class="sortable-item <?= $field['is_active'] ? 'active-field' : 'inactive-field' ?>" 
                            data-id="<?= $field['id'] ?>">
                            <div class="d-flex justify-content-between align-items-center">
                                <div>
                                    <strong><?= htmlspecialchars($field['field_name']) ?></strong>
                                    <span class="badge badge-info field-type-badge ml-2">
                                        <?= $field['field_type'] ?>
                                    </span>
                                    <?php if($field['unit_label']): ?>
                                        <span class="unit-label ml-2">(<?= $field['unit_label'] ?>)</span>
                                    <?php endif; ?>
                                    <?php if($field['is_required']): ?>
                                        <span class="badge badge-warning ml-1">Required</span>
                                    <?php endif; ?>
                                    <?php if(!$field['is_active']): ?>
                                        <span class="badge badge-secondary ml-1">Inactive</span>
                                    <?php endif; ?>
                                </div>
                                <div>
                                    <button class="btn btn-sm btn-info" data-toggle="modal" 
                                            data-target="#editFieldModal<?= $field['id'] ?>">
                                        Edit
                                    </button>
                                    <?php if($field['is_active']): ?>
                                        <form method="post" class="d-inline">
                                            <input type="hidden" name="field_id" value="<?= $field['id'] ?>">
                                            <button name="delete_field" class="btn btn-sm btn-danger"
                                                    onclick="return confirm('Delete this field?')">
                                                Delete
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <!-- Edit Modal -->
                            <div class="modal fade" id="editFieldModal<?= $field['id'] ?>" tabindex="-1">
                                <div class="modal-dialog">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title">Edit Field</h5>
                                        </div>
                                        <form method="post">
                                            <div class="modal-body">
                                                <input type="hidden" name="field_id" value="<?= $field['id'] ?>">
                                                
                                                <div class="form-group">
                                                    <label>Field Name</label>
                                                    <input type="text" name="field_name" 
                                                           value="<?= htmlspecialchars($field['field_name']) ?>" 
                                                           class="form-control" required>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label>Field Type</label>
                                                    <select name="field_type" class="form-control" required>
                                                        <?php 
                                                        $types = ['hours', 'dollars', 'miles', 'units', 'percentage', 'number', 'text'];
                                                        foreach($types as $type): ?>
                                                            <option value="<?= $type ?>" 
                                                                <?= $field['field_type'] == $type ? 'selected' : '' ?>>
                                                                <?= ucfirst($type) ?>
                                                            </option>
                                                        <?php endforeach; ?>
                                                    </select>
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label>Unit Label</label>
                                                    <input type="text" name="unit_label" 
                                                           value="<?= htmlspecialchars($field['unit_label']) ?>" 
                                                           class="form-control">
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label>Default Value</label>
                                                    <input type="number" step="0.01" name="default_value" 
                                                           value="<?= $field['default_value'] ?>" 
                                                           class="form-control">
                                                </div>
                                                
                                                <div class="form-group">
                                                    <label>Display Order</label>
                                                    <input type="number" name="display_order" 
                                                           value="<?= $field['display_order'] ?>" 
                                                           class="form-control">
                                                </div>
                                                
                                                <div class="form-check mb-2">
                                                    <input type="checkbox" name="is_required" 
                                                           class="form-check-input" 
                                                           <?= $field['is_required'] ? 'checked' : '' ?>>
                                                    <label class="form-check-label">Required Field</label>
                                                </div>
                                                
                                                <div class="form-check">
                                                    <input type="checkbox" name="is_active" 
                                                           class="form-check-input" 
                                                           <?= $field['is_active'] ? 'checked' : '' ?>>
                                                    <label class="form-check-label">Active</label>
                                                </div>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" 
                                                        data-dismiss="modal">Cancel</button>
                                                <button name="update_field" class="btn btn-primary">Save</button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </li>
                    <?php endwhile; ?>
                </ul>
            </div>
        </div>
    </div>
    
    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/ui/1.12.1/jquery-ui.min.js"></script>
    <script>
        // Make fields sortable
        $(function() {
            $("#sortableFields").sortable({
                update: function(event, ui) {
                    var order = [];
                    $("#sortableFields .sortable-item").each(function(index) {
                        order.push({
                            id: $(this).data('id'),
                            position: index + 1
                        });
                    });
                    
                    // Save order to database
                    $.post('update_field_order.php', { order: order }, function(response) {
                        console.log('Order updated');
                    });
                }
            });
        });
    </script>
</body>
</html>