<?php
session_start();
if (!isset($_SESSION['SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';

// Guru API Configuration
define('GURU_CLIENT_ID', 'your_guru_client_id');
define('GURU_CLIENT_SECRET', 'your_guru_client_secret');
define('GURU_REDIRECT_URI', 'https://yourdomain.com/app/guru_callback.php');
define('GURU_BASE_URL', 'https://api.getguru.com/api/v1');
define('GURU_AUTH_URL', 'https://api.getguru.com/oauth/authorize');
define('GURU_TOKEN_URL', 'https://api.getguru.com/oauth/token');

// Ensure tables exist
ensureGuruTables();

function ensureGuruTables() {
    global $con;
    
    $tables = [
        "CREATE TABLE IF NOT EXISTS guru_tokens (
            id INT AUTO_INCREMENT PRIMARY KEY,
            access_token TEXT NOT NULL,
            refresh_token TEXT NOT NULL,
            user_id VARCHAR(100),
            email VARCHAR(255),
            expires_at DATETIME NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )"
    ];
    
    foreach ($tables as $sql) {
        mysqli_query($con, $sql);
    }
}

// Get Guru access token
function getGuruAccessToken() {
    global $con;
    
    $query = "SELECT * FROM guru_tokens ORDER BY id DESC LIMIT 1";
    $result = mysqli_query($con, $query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $token = mysqli_fetch_assoc($result);
        
        // Check if token is expired
        $expires_time = strtotime($token['expires_at']);
        $current_time = time();
        
        if ($current_time < $expires_time - 300) {
            return $token;
        }
        
        // Token expired, refresh it
        return refreshGuruToken($token['refresh_token']);
    }
    
    return null;
}

// Refresh Guru token
function refreshGuruToken($refresh_token) {
    $ch = curl_init(GURU_TOKEN_URL);
    
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query([
            'grant_type' => 'refresh_token',
            'refresh_token' => $refresh_token,
            'client_id' => GURU_CLIENT_ID,
            'client_secret' => GURU_CLIENT_SECRET
        ]),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/x-www-form-urlencoded',
            'Accept: application/json'
        ]
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        $data = json_decode($response, true);
        
        // Save new token
        global $con;
        $expires_at = date('Y-m-d H:i:s', time() + $data['expires_in']);
        
        $stmt = mysqli_prepare($con, "
            INSERT INTO guru_tokens 
            (access_token, refresh_token, user_id, email, expires_at) 
            VALUES (?, ?, ?, ?, ?)
        ");
        
        mysqli_stmt_bind_param($stmt, "sssss", 
            $data['access_token'], 
            $data['refresh_token'], 
            $data['user_id'] ?? null,
            $data['email'] ?? null,
            $expires_at
        );
        
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        
        logSyncActivity('guru', 'token_refresh', 'success', 'Token refreshed successfully');
        
        return [
            'access_token' => $data['access_token'],
            'refresh_token' => $data['refresh_token'],
            'user_id' => $data['user_id'] ?? null,
            'email' => $data['email'] ?? null,
            'expires_at' => $expires_at
        ];
    }
    
    logSyncActivity('guru', 'token_refresh', 'error', "Failed to refresh token: HTTP $http_code");
    return null;
}

// Guru API Request
function guru_request($method, $endpoint, $data = null) {
    $token = getGuruAccessToken();
    
    if (!$token) {
        return ['error' => 'No valid token', 'http_code' => 401];
    }
    
    $url = GURU_BASE_URL . $endpoint;
    
    $ch = curl_init($url);
    $headers = [
        'Authorization: Bearer ' . $token['access_token'],
        'Accept: application/json',
        'Content-Type: application/json'
    ];
    
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST => $method,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30
    ]);
    
    if ($data) {
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return [
        'http_code' => $http_code,
        'response' => json_decode($response, true),
        'raw_response' => $response
    ];
}

// Sync employee to Guru
function syncEmployeeToGuru($employee_id) {
    global $con;
    
    // Get employee data
    $query = "SELECT u.*, im.external_id as guru_user_id 
              FROM usermst u 
              LEFT JOIN integration_mappings im ON im.local_employee_id = u.id 
                AND im.integration_type = 'guru'
              WHERE u.id = $employee_id";
    
    $result = mysqli_query($con, $query);
    if (!$result || mysqli_num_rows($result) == 0) {
        return ['success' => false, 'error' => 'Employee not found'];
    }
    
    $employee = mysqli_fetch_assoc($result);
    
    // Check if user exists in Guru
    $search_response = guru_request('GET', '/users?email=' . urlencode($employee['vEmail']));
    
    $guru_user_id = null;
    
    if ($search_response['http_code'] == 200 && !empty($search_response['response'])) {
        // User exists, get ID
        $guru_user_id = $search_response['response'][0]['id'] ?? null;
    } else {
        // Create new user in Guru
        $user_data = [
            'email' => $employee['vEmail'],
            'firstName' => explode(' ', $employee['vName'])[0] ?? '',
            'lastName' => explode(' ', $employee['vName'])[1] ?? '',
            'role' => 'MEMBER'
        ];
        
        $create_response = guru_request('POST', '/users', $user_data);
        
        if ($create_response['http_code'] == 200 || $create_response['http_code'] == 201) {
            $guru_user_id = $create_response['response']['id'] ?? null;
        }
    }
    
    if ($guru_user_id) {
        // Update or create mapping
        if ($employee['guru_user_id']) {
            mysqli_query($con, "
                UPDATE integration_mappings 
                SET external_id = '$guru_user_id',
                    last_sync_at = NOW()
                WHERE local_employee_id = $employee_id 
                AND integration_type = 'guru'
            ");
        } else {
            mysqli_query($con, "
                INSERT INTO integration_mappings 
                (integration_type, local_employee_id, external_id, external_name, sync_direction)
                VALUES ('guru', $employee_id, '$guru_user_id', '{$employee['vName']}', 'both')
            ");
        }
        
        // Update usermst with Guru user ID
        mysqli_query($con, "
            UPDATE usermst 
            SET guru_user_id = '$guru_user_id'
            WHERE id = $employee_id
        ");
        
        logSyncActivity('guru', 'user_sync', 'success', 
            "Employee {$employee['vName']} synced to Guru", 
            'employee', $employee_id, null, $search_response['response']);
        
        return ['success' => true, 'guru_user_id' => $guru_user_id];
    }
    
    logSyncActivity('guru', 'user_sync', 'error', 
        "Failed to sync employee {$employee['vName']} to Guru", 
        'employee', $employee_id, null, $search_response);
    
    return ['success' => false, 'error' => 'Failed to sync to Guru'];
}

// Create Guru card from timesheet
function createGuruCard($timesheet_data) {
    global $con;
    
    $employee_id = $timesheet_data['employee_id'];
    $week_start = $timesheet_data['week_start'];
    
    // Get employee Guru mapping
    $query = "SELECT u.vName, u.guru_user_id, im.external_id
              FROM usermst u
              LEFT JOIN integration_mappings im ON im.local_employee_id = u.id 
                AND im.integration_type = 'guru'
              WHERE u.id = $employee_id";
    
    $result = mysqli_query($con, $query);
    if (!$result || mysqli_num_rows($result) == 0) {
        return ['success' => false, 'error' => 'Employee not found'];
    }
    
    $employee = mysqli_fetch_assoc($result);
    
    // Sync employee if not already synced
    if (!$employee['guru_user_id'] && !$employee['external_id']) {
        $sync_result = syncEmployeeToGuru($employee_id);
        if (!$sync_result['success']) {
            return $sync_result;
        }
        $guru_user_id = $sync_result['guru_user_id'];
    } else {
        $guru_user_id = $employee['guru_user_id'] ?? $employee['external_id'];
    }
    
    // Create card content
    $content = "
        ## Timesheet Summary - {$employee['vName']}
        **Week:** {$week_start}
        
        ### Hours Breakdown:
        - Regular Hours: {$timesheet_data['hours_total']}
        - Solo Hours: {$timesheet_data['solo_hours']}
        - Total Hours: " . ($timesheet_data['hours_total'] + $timesheet_data['solo_hours']) . "
        
        ### Additional:
        - Miles: {$timesheet_data['miles']}
        - Miscellaneous: {$timesheet_data['miscellaneous']}
        
        ### Pay Summary:
        - Regular Pay: \${$timesheet_data['regular_pay']}
        - Solo Bonus: \${$timesheet_data['solo_bonus_pay']}
        - Mileage: \${$timesheet_data['mileage_pay']}
        - **Total Gross: \${$timesheet_data['gross_pay']}**
        
        ### Custom Fields:
    ";
    
    if (!empty($timesheet_data['custom_fields'])) {
        foreach ($timesheet_data['custom_fields'] as $field) {
            $content .= "- {$field['field_name']}: {$field['field_value']} {$field['unit_label']} = \${$field['calculated_pay']}\n";
        }
    }
    
    $content .= "
        
        ---
        *Automatically generated from MaidToSparkle payroll system*
    ";
    
    // Create card data
    $card_data = [
        'content' => $content,
        'preferredPhraseology' => "Timesheet: {$employee['vName']} - {$week_start}",
        'collection' => [
            'id' => 'timesheets-collection-id' // Replace with actual collection ID
        ],
        'shareStatus' => 'TEAM',
        'owner' => [
            'id' => $guru_user_id
        ],
        'tags' => ['timesheet', 'payroll', $week_start],
        'verification' => [
            'type' => 'NONE'
        ]
    ];
    
    $response = guru_request('POST', '/cards', $card_data);
    
    if ($response['http_code'] == 200 || $response['http_code'] == 201) {
        $card_id = $response['response']['id'] ?? null;
        
        logSyncActivity('guru', 'card_create', 'success', 
            "Card created for {$employee['vName']} - {$week_start}", 
            'timesheet', $timesheet_data['id'], $card_data, $response['response']);
        
        return ['success' => true, 'card_id' => $card_id];
    }
    
    logSyncActivity('guru', 'card_create', 'error', 
        "Failed to create card for {$employee['vName']}", 
        'timesheet', $timesheet_data['id'], $card_data, $response);
    
    return ['success' => false, 'error' => 'API Error', 'response' => $response];
}

// Sync timesheet to Guru
function syncTimesheetToGuru($report_id) {
    global $con;
    
    // Get report data with custom fields
    $query = "
        SELECT wr.*, u.vName,
               (SELECT GROUP_CONCAT(CONCAT(cpf.field_name, ':', wrf.field_value, ':', wrf.calculated_pay) SEPARATOR '|')
                FROM weekly_report_custom_fields wrf
                JOIN custom_payroll_fields cpf ON cpf.id = wrf.custom_field_id
                WHERE wrf.weekly_report_id = wr.id) as custom_fields_data
        FROM weekly_reports wr
        JOIN usermst u ON u.id = wr.employee_id
        WHERE wr.id = $report_id
    ";
    
    $result = mysqli_query($con, $query);
    if (!$result || mysqli_num_rows($result) == 0) {
        return ['success' => false, 'error' => 'Report not found'];
    }
    
    $report = mysqli_fetch_assoc($con);
    
    // Parse custom fields data
    $custom_fields = [];
    if (!empty($report['custom_fields_data'])) {
        $fields = explode('|', $report['custom_fields_data']);
        foreach ($fields as $field) {
            list($name, $value, $pay) = explode(':', $field);
            $custom_fields[] = [
                'field_name' => $name,
                'field_value' => $value,
                'calculated_pay' => $pay
            ];
        }
    }
    
    // Prepare timesheet data
    $timesheet_data = [
        'id' => $report['id'],
        'employee_id' => $report['employee_id'],
        'week_start' => $report['week_start'],
        'hours_total' => $report['hours_total'],
        'solo_hours' => $report['solo_hours'],
        'miles' => $report['miles'],
        'miscellaneous' => $report['miscellaneous'],
        'regular_pay' => $report['regular_pay'],
        'solo_bonus_pay' => $report['solo_bonus_pay'],
        'mileage_pay' => $report['mileage_pay'],
        'gross_pay' => $report['gross_pay'],
        'custom_fields' => $custom_fields
    ];
    
    // Create Guru card
    $result = createGuruCard($timesheet_data);
    
    if ($result['success']) {
        // Mark as synced
        mysqli_query($con, "
            UPDATE weekly_reports 
            SET guru_synced = 1, 
                guru_ref = '{$result['card_id']}',
                updated_at = NOW()
            WHERE id = $report_id
        ");
        
        return ['success' => true, 'card_id' => $result['card_id']];
    }
    
    return ['success' => false, 'error' => $result['error']];
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['connect_guru'])) {
        // Start OAuth flow
        $params = [
            'client_id' => GURU_CLIENT_ID,
            'response_type' => 'code',
            'redirect_uri' => GURU_REDIRECT_URI,
            'scope' => 'read write',
            'state' => bin2hex(random_bytes(16))
        ];
        
        $_SESSION['guru_state'] = $params['state'];
        $auth_url = GURU_AUTH_URL . '?' . http_build_query($params);
        
        header('Location: ' . $auth_url);
        exit;
    }
    
    if (isset($_POST['sync_timesheet'])) {
        $report_id = intval($_POST['report_id']);
        $result = syncTimesheetToGuru($report_id);
        
        $_SESSION['guru_sync_result'] = $result;
        header('Location: guru_integration.php');
        exit;
    }
    
    if (isset($_POST['sync_all_users'])) {
        // Sync all employees to Guru
        $employees = mysqli_query($con, "
            SELECT id, vName FROM usermst 
            WHERE is_active = 1 
            AND vEmail IS NOT NULL
        ");
        
        $results = [];
        while ($emp = mysqli_fetch_assoc($employees)) {
            $result = syncEmployeeToGuru($emp['id']);
            $results[] = [
                'employee' => $emp['vName'],
                'success' => $result['success'],
                'guru_user_id' => $result['guru_user_id'] ?? null,
                'error' => $result['error'] ?? null
            ];
        }
        
        $_SESSION['guru_user_sync_results'] = $results;
        header('Location: guru_integration.php');
        exit;
    }
}

// Get connection status
$token_status = getGuruAccessToken();
$connected = $token_status ? true : false;

// Get unsynced reports
$unsynced_reports = mysqli_query($con, "
    SELECT wr.id, wr.week_start, wr.hours_total, wr.gross_pay, u.vName
    FROM weekly_reports wr
    JOIN usermst u ON u.id = wr.employee_id
    WHERE wr.guru_synced = 0
    AND wr.gusto_submitted = 1
    ORDER BY wr.week_start DESC
    LIMIT 20
");

// Get sync logs
$sync_logs = mysqli_query($con, "
    SELECT * FROM integration_sync_logs 
    WHERE integration_type = 'guru'
    ORDER BY created_at DESC
    LIMIT 20
");
?>

<!DOCTYPE html>
<html>
<head>
    <title>Guru Integration</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <link href="css/font-awesome.css" rel="stylesheet">
    <style>
        .guru-card { border-left: 4px solid #6f42c1; }
        .report-item { border-bottom: 1px solid #dee2e6; padding: 10px 0; }
        .report-item:last-child { border-bottom: none; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h3>Guru Integration</h3>
        
        <!-- Connection Status -->
        <div class="card mb-4 guru-card">
            <div class="card-header <?= $connected ? 'bg-success' : 'bg-danger' ?> text-white">
                <h5 class="mb-0">
                    <i class="fa fa-<?= $connected ? 'check-circle' : 'times-circle' ?>"></i>
                    Connection Status: <?= $connected ? 'Connected' : 'Disconnected' ?>
                </h5>
            </div>
            <div class="card-body">
                <?php if($connected): ?>
                    <div class="alert alert-success">
                        <h5><i class="fa fa-check"></i> Connected to Guru</h5>
                        <p class="mb-0">
                            Connected as: <?= $token_status['email'] ?? 'Unknown' ?>
                            <?php if($token_status['user_id']): ?>
                                (User ID: <?= $token_status['user_id'] ?>)
                            <?php endif; ?>
                        </p>
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>Sync Users</h6>
                                    <p>Sync employees as Guru users</p>
                                    <form method="post">
                                        <button name="sync_all_users" class="btn btn-primary">
                                            <i class="fa fa-users"></i> Sync All Employees
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>Test Connection</h6>
                                    <p>Test Guru API connection</p>
                                    <a href="test_guru.php" class="btn btn-info">
                                        <i class="fa fa-plug"></i> Test Connection
                                    </a>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>View Cards</h6>
                                    <p>View created Guru cards</p>
                                    <a href="guru_cards.php" class="btn btn-secondary">
                                        <i class="fa fa-eye"></i> View Cards
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Sync Results -->
                    <?php if(isset($_SESSION['guru_sync_result'])): 
                        $result = $_SESSION['guru_sync_result'];
                    ?>
                        <div class="alert alert-<?= $result['success'] ? 'success' : 'danger' ?>">
                            <h6>Sync Result:</h6>
                            <?= $result['success'] 
                                ? 'Timesheet synced successfully to Guru (Card ID: ' . $result['card_id'] . ')' 
                                : 'Error: ' . ($result['error'] ?? 'Unknown error') ?>
                        </div>
                        <?php unset($_SESSION['guru_sync_result']); ?>
                    <?php endif; ?>
                    
                    <?php if(isset($_SESSION['guru_user_sync_results'])): ?>
                        <div class="card mb-4">
                            <div class="card-header bg-info text-white">
                                <h6>User Sync Results</h6>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Employee</th>
                                                <th>Status</th>
                                                <th>Guru User ID</th>
                                                <th>Error</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach($_SESSION['guru_user_sync_results'] as $result): ?>
                                                <tr class="<?= $result['success'] ? 'table-success' : 'table-danger' ?>">
                                                    <td><?= $result['employee'] ?></td>
                                                    <td>
                                                        <?= $result['success'] 
                                                            ? '<span class="badge badge-success">Success</span>' 
                                                            : '<span class="badge badge-danger">Failed</span>' ?>
                                                    </td>
                                                    <td><?= $result['guru_user_id'] ?? '-' ?></td>
                                                    <td><?= $result['error'] ?? '-' ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                        <?php unset($_SESSION['guru_user_sync_results']); ?>
                    <?php endif; ?>
                    
                    <!-- Unsynced Reports -->
                    <div class="card">
                        <div class="card-header bg-warning">
                            <h6 class="mb-0">Unsynced Timesheets (<?= mysqli_num_rows($unsynced_reports) ?>)</h6>
                        </div>
                        <div class="card-body">
                            <?php if(mysqli_num_rows($unsynced_reports) > 0): ?>
                                <?php while($report = mysqli_fetch_assoc($unsynced_reports)): ?>
                                    <div class="report-item d-flex justify-content-between align-items-center">
                                        <div>
                                            <strong><?= $report['vName'] ?></strong>
                                            <small class="text-muted"> | Week: <?= $report['week_start'] ?></small>
                                            <div class="small">
                                                Hours: <?= $report['hours_total'] ?> | 
                                                Gross: $<?= number_format($report['gross_pay'], 2) ?>
                                            </div>
                                        </div>
                                        <form method="post" class="mb-0">
                                            <input type="hidden" name="report_id" value="<?= $report['id'] ?>">
                                            <button name="sync_timesheet" class="btn btn-sm btn-primary">
                                                <i class="fa fa-sync"></i> Sync to Guru
                                            </button>
                                        </form>
                                    </div>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <div class="alert alert-info">
                                    All timesheets are synced with Guru.
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                <?php else: ?>
                    <div class="alert alert-warning">
                        <h5><i class="fa fa-exclamation-triangle"></i> Not Connected</h5>
                        <p class="mb-3">Connect to Guru to sync timesheet data</p>
                        <form method="post">
                            <button name="connect_guru" class="btn btn-success btn-lg">
                                <i class="fa fa-link"></i> Connect to Guru
                            </button>
                        </form>
                    </div>
                    
                    <div class="alert alert-info">
                        <h6><i class="fa fa-info-circle"></i> About Guru Integration:</h6>
                        <p class="mb-0">
                            Guru integration allows you to automatically create cards in Guru 
                            for each timesheet. This helps keep your team informed about 
                            payroll and hours worked.
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Sync Logs -->
        <div class="card">
            <div class="card-header bg-light">
                <h6 class="mb-0"><i class="fa fa-history"></i> Recent Sync Logs</h6>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Time</th>
                                <th>Action</th>
                                <th>Entity</th>
                                <th>Status</th>
                                <th>Message</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while($log = mysqli_fetch_assoc($sync_logs)): ?>
                                <tr class="table-<?= $log['status'] == 'success' ? 'success' : ($log['status'] == 'error' ? 'danger' : 'warning') ?>">
                                    <td><?= date('Y-m-d H:i', strtotime($log['created_at'])) ?></td>
                                    <td><?= $log['action_type'] ?></td>
                                    <td><?= $log['entity_type'] ?> <?= $log['entity_id'] ?></td>
                                    <td>
                                        <span class="badge badge-<?= $log['status'] ?>">
                                            <?= $log['status'] ?>
                                        </span>
                                    </td>
                                    <td><?= substr($log['message'], 0, 80) ?>...</td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <script src="js/jquery-1.12.4.js"></script>
    <script src="js/bootstrap.min.js"></script>
</body>
</html>