<?php
// Turn on error reporting
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

// Include db first
require_once 'db.php';

// Check login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['SESS_MEMBER_ID'])) {
    header('Location: index.php');
    exit;
}

// Store flash messages before any output
$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
$warning_message = $_SESSION['warning'] ?? '';

// Clear flash messages after storing
unset($_SESSION['success']);
unset($_SESSION['error']);
unset($_SESSION['warning']);

// Include Gusto files
require_once 'gusto_api.php';
$config = require __DIR__ . '/gusto_config.php';

// Function to get state tax rate
function getStateTaxRate($state) {
    $state = strtoupper($state);
    
    // States with no income tax
    $no_tax_states = ['AK', 'FL', 'NV', 'NH', 'SD', 'TN', 'TX', 'WA', 'WY'];
    if (in_array($state, $no_tax_states)) {
        return 0.00;
    }
    
    // 2024 state tax rates
    $tax_rates = [
        'AL' => 0.0400, 'AZ' => 0.0250, 'AR' => 0.0550, 'CA' => 0.0600,
        'CO' => 0.0455, 'CT' => 0.0500, 'DE' => 0.0550, 'GA' => 0.0500,
        'HI' => 0.0700, 'ID' => 0.0500, 'IL' => 0.0495, 'IN' => 0.0323,
        'IA' => 0.0600, 'KS' => 0.0500, 'KY' => 0.0450, 'LA' => 0.0400,
        'ME' => 0.0650, 'MD' => 0.0475, 'MA' => 0.0500, 'MI' => 0.0425,
        'MN' => 0.0685, 'MS' => 0.0400, 'MO' => 0.0500, 'MT' => 0.0500,
        'NE' => 0.0500, 'NJ' => 0.0500, 'NM' => 0.0450, 'NY' => 0.0550,
        'NC' => 0.0475, 'ND' => 0.0150, 'OH' => 0.0300, 'OK' => 0.0450,
        'OR' => 0.0800, 'PA' => 0.0307, 'RI' => 0.0450, 'SC' => 0.0500,
        'VT' => 0.0600, 'VA' => 0.0525, 'WV' => 0.0500, 'WI' => 0.0550,
        'UT' => 0.0485
    ];
    
    return $tax_rates[$state] ?? 0.0500;
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['calculate_payroll'])) {
        $payroll_uuid = $_POST['payroll_uuid'];
        
        $calculate_response = gusto_request('PUT', "companies/{$config->COMPANY_UUID}/payrolls/{$payroll_uuid}/calculate");
        
        if ($calculate_response['http_code'] === 202) {
            $_SESSION['success'] = "✅ Payroll calculation started! This may take a few minutes.";
            $_SESSION['calculation_started'] = time();
            $_SESSION['calculating_payroll'] = $payroll_uuid;
        } elseif ($calculate_response['http_code'] === 200) {
            $_SESSION['success'] = "✅ Payroll calculated successfully!";
        } else {
            $_SESSION['error'] = "Failed to calculate payroll. HTTP " . $calculate_response['http_code'];
        }
        
        header('Location: gusto_payroll_reports.php?payroll_uuid=' . $payroll_uuid);
        exit;
    }
    
    if (isset($_POST['refresh_status'])) {
        $payroll_uuid = $_POST['payroll_uuid'];
        header('Location: gusto_payroll_reports.php?payroll_uuid=' . $payroll_uuid);
        exit;
    }
    
    // Handle dropdown selection - ONLY when View Report is clicked
    if (isset($_POST['select_payroll'])) {
        $payroll_uuid = $_POST['payroll_uuid'];
        if (!empty($payroll_uuid)) {
            header('Location: gusto_payroll_reports.php?payroll_uuid=' . $payroll_uuid);
            exit;
        } else {
            $_SESSION['error'] = "Please select a payroll from the dropdown.";
            header('Location: gusto_payroll_reports.php');
            exit;
        }
    }
}

// Fetch payrolls from BOTH payrolls table AND weekly_reports (to catch all submitted payrolls)
$payrolls = [];

// First, get payrolls from payrolls table
$payrolls_result = mysqli_query($con, "
    SELECT p.*, 
           (SELECT COUNT(*) FROM weekly_reports wr WHERE wr.gusto_payroll_uuid = p.payroll_uuid) as report_count 
    FROM payrolls p 
    ORDER BY p.created_at DESC 
    LIMIT 20
");

if ($payrolls_result) {
    while ($row = mysqli_fetch_assoc($payrolls_result)) {
        $payrolls[$row['payroll_uuid']] = $row;
    }
}

// Then, get payrolls from weekly_reports that might not be in payrolls table
$weekly_reports_payrolls = mysqli_query($con, "
    SELECT DISTINCT gusto_payroll_uuid, week_start 
    FROM weekly_reports 
    WHERE gusto_payroll_uuid IS NOT NULL AND gusto_payroll_uuid != '' 
    ORDER BY week_start DESC 
    LIMIT 20
");

if ($weekly_reports_payrolls) {
    while ($row = mysqli_fetch_assoc($weekly_reports_payrolls)) {
        $payroll_uuid = $row['gusto_payroll_uuid'];
        $week_start = $row['week_start'];
        
        // If this payroll isn't already in our list, add it
        if (!isset($payrolls[$payroll_uuid])) {
            // Count reports for this payroll
            $report_count_result = mysqli_query($con, "
                SELECT COUNT(*) as report_count 
                FROM weekly_reports 
                WHERE gusto_payroll_uuid = '" . mysqli_real_escape_string($con, $payroll_uuid) . "'
            ");
            $report_count_data = mysqli_fetch_assoc($report_count_result);
            $report_count = $report_count_data['report_count'];
            
            $payrolls[$payroll_uuid] = [
                'payroll_uuid' => $payroll_uuid,
                'week_start' => $week_start,
                'report_count' => $report_count,
                'status' => 'submitted',
                'pay_date' => date('Y-m-d', strtotime($week_start . ' +11 days')),
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
    }
}

// Convert associative array back to indexed array for display
$payrolls = array_values($payrolls);

// Sort by week_start descending
usort($payrolls, function($a, $b) {
    return strtotime($b['week_start']) - strtotime($a['week_start']);
});

// If no payrolls found at all
if (empty($payrolls)) {
    $no_payrolls_message = "No payrolls found. Submit a week from the Weekly Reports page first.";
}

$selected_payroll = null;
$payroll_details = [];
$payroll_reports = [];
$employee_compensations = [];
$error = '';
$calculation_in_progress = false;

// Handle payroll selection from URL parameter
if (isset($_GET['payroll_uuid']) && !empty($_GET['payroll_uuid'])) {
    $selected_payroll_uuid = mysqli_real_escape_string($con, $_GET['payroll_uuid']);
    $selected_payroll = $selected_payroll_uuid;
    
    // DEBUG: Check if payroll exists in our list
    $payroll_exists = false;
    foreach ($payrolls as $payroll) {
        if ($payroll['payroll_uuid'] === $selected_payroll_uuid) {
            $payroll_exists = true;
            break;
        }
    }
    
    if (!$payroll_exists) {
        $_SESSION['error'] = "Payroll not found: " . substr($selected_payroll_uuid, 0, 8) . "...";
        header('Location: gusto_payroll_reports.php');
        exit;
    }
    
    // Check if we're waiting for calculation
    if (isset($_SESSION['calculating_payroll']) && $_SESSION['calculating_payroll'] === $selected_payroll_uuid) {
        $calculation_in_progress = true;
        $time_elapsed = time() - $_SESSION['calculation_started'];
        if ($time_elapsed > 300) {
            unset($_SESSION['calculating_payroll']);
            unset($_SESSION['calculation_started']);
            $_SESSION['error'] = "Calculation timed out after 5 minutes. Please try again.";
        }
    }
    
    // Fetch payroll details from Gusto API
    $api_response = gusto_request('GET', "companies/{$config->COMPANY_UUID}/payrolls/{$selected_payroll_uuid}");
    
    if ($api_response['http_code'] === 200 && !empty($api_response['json'])) {
        $payroll_details = $api_response['json'];
        
        // If payroll is now calculated, clear calculation session
        if (!empty($payroll_details['calculated_at']) && isset($_SESSION['calculating_payroll'])) {
            unset($_SESSION['calculating_payroll']);
            unset($_SESSION['calculation_started']);
            $_SESSION['success'] = "✅ Payroll calculation completed successfully!";
        }
        
    } else {
        $error = "Gusto API returned HTTP " . $api_response['http_code'] . ". Payroll may still be processing.";
    }
    
    // Fetch payroll info from our database (both payrolls table and weekly_reports)
    $payroll_info = mysqli_fetch_assoc(mysqli_query($con, "SELECT * FROM payrolls WHERE payroll_uuid = '" . mysqli_real_escape_string($con, $selected_payroll_uuid) . "'"));
    
    // If not in payrolls table, get info from weekly_reports
    if (!$payroll_info) {
        $weekly_info = mysqli_fetch_assoc(mysqli_query($con, "
            SELECT week_start, MAX(created_at) as created_at 
            FROM weekly_reports 
            WHERE gusto_payroll_uuid = '" . mysqli_real_escape_string($con, $selected_payroll_uuid) . "' 
            LIMIT 1
        "));
        
        if ($weekly_info) {
            $payroll_info = [
                'week_start' => $weekly_info['week_start'],
                'pay_date' => date('Y-m-d', strtotime($weekly_info['week_start'] . ' +11 days')),
                'status' => 'submitted',
                'created_at' => $weekly_info['created_at']
            ];
        }
    }
    
    // If not in database, create info from Gusto data
    if (!$payroll_info && !empty($payroll_details)) {
        $payroll_info = [
            'week_start' => $payroll_details['pay_period']['start_date'] ?? 'N/A',
            'pay_date' => $payroll_details['check_date'] ?? 'N/A',
            'status' => $payroll_details['processed'] ? 'processed' : ($payroll_details['calculated_at'] ? 'calculated' : 'created')
        ];
    }
    
    // Fetch associated reports from our database with pay breakdown
    $reports_query = mysqli_query($con, "
        SELECT wr.*, u.vName as employee_name, u.gusto_employee_uuid, u.state, u.home_address_state
        FROM weekly_reports wr 
        JOIN usermst u ON u.id = wr.employee_id 
        WHERE wr.gusto_payroll_uuid = '" . mysqli_real_escape_string($con, $selected_payroll_uuid) . "'
        ORDER BY u.vName
    ");
    
    if ($reports_query) {
        while ($report = mysqli_fetch_assoc($reports_query)) {
            $payroll_reports[] = $report;
        }
    }
    
    // Create compensation data if we have reports (with detailed breakdown INCLUDING miscellaneous)
    if (!empty($payroll_reports)) {
        foreach ($payroll_reports as $report) {
            // Use pre-calculated values from weekly_reports table - FIXED: These should now be correct
            $regular_pay = $report['regular_pay'] ?? 0;
            $solo_bonus = $report['solo_bonus_pay'] ?? 0;
            $mileage_pay = $report['mileage_pay'] ?? 0;
            $miscellaneous_pay = $report['miscellaneous_pay'] ?? 0;
            $gross_pay = $report['gross_pay'] ?? 0;
            
            // Get employee state for tax calculation
            $employee_state = $report['state'] ?? $report['home_address_state'] ?? 'CA';
            
            // Calculate taxes
            $federal_tax = $gross_pay * 0.12;
            $social_security = $gross_pay * 0.062;
            $medicare = $gross_pay * 0.0145;
            $state_tax_rate = getStateTaxRate($employee_state);
            $state_tax = $gross_pay * $state_tax_rate;
            $total_taxes = $federal_tax + $social_security + $medicare + $state_tax;
            $net_pay = $gross_pay - $total_taxes;
            
            $employee_compensations[] = [
                'employee' => [
                    'name' => $report['employee_name'],
                    'uuid' => $report['gusto_employee_uuid'] ?? 'simulated'
                ],
                'gross_pay' => $gross_pay,
                'net_pay' => $net_pay,
                'hours' => $report['hours_total'],
                'payment_method' => 'Direct Deposit',
                'pay_breakdown' => [
                    'regular_hours' => $report['hours_total'], // FIXED: All hours are regular hours
                    'regular_pay' => $regular_pay,
                    'solo_hours' => $report['solo_hours'],
                    'solo_bonus' => $solo_bonus,
                    'mileage' => $report['miles'],
                    'mileage_pay' => $mileage_pay,
                    'miscellaneous' => $report['miscellaneous'] ?? 0,
                    'miscellaneous_pay' => $miscellaneous_pay
                ],
                'employee_taxes' => [
                    'federal_income_tax' => $federal_tax,
                    'social_security_tax' => $social_security,
                    'medicare_tax' => $medicare,
                    'state_income_tax' => $state_tax,
                    'total' => $total_taxes
                ],
                'employee_state' => $employee_state,
                'state_tax_rate' => $state_tax_rate
            ];
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>MaidToSparkle: Payroll Reports</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <style>
        /* Flash Message Styles - Big and Centered - FIXED */
        .flash-message-container {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 9999;
            width: 90%;
            max-width: 600px;
            animation: flashMessage 0.5s ease-out;
        }
        
        .flash-message {
            font-size: 1.5rem;
            font-weight: bold;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            text-align: center;
            border: none;
            margin-bottom: 20px;
            position: relative;
        }
        
        .flash-success {
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            border-left: 10px solid #1e7e34;
        }
        
        .flash-error {
            background: linear-gradient(135deg, #dc3545, #e83e8c);
            color: white;
            border-left: 10px solid #bd2130;
        }
        
        .flash-warning {
            background: linear-gradient(135deg, #ffc107, #fd7e14);
            color: white;
            border-left: 10px solid #d39e00;
        }
        
        .flash-icon {
            font-size: 3rem;
            margin-bottom: 15px;
            display: block;
        }
        
        .flash-close {
            background: rgba(255,255,255,0.2);
            border: none;
            color: white;
            font-size: 1.5rem;
            padding: 5px 15px;
            border-radius: 50%;
            position: absolute;
            top: 10px;
            right: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
            z-index: 10000;
        }
        
        .flash-close:hover {
            background: rgba(255,255,255,0.3);
            transform: scale(1.1);
        }
        
        @keyframes flashMessage {
            0% {
                opacity: 0;
                transform: translate(-50%, -60%) scale(0.8);
            }
            50% {
                opacity: 1;
                transform: translate(-50%, -50%) scale(1.05);
            }
            100% {
                opacity: 1;
                transform: translate(-50%, -50%) scale(1);
            }
        }
        
        .flash-backdrop {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.7);
            z-index: 9998;
            animation: fadeIn 0.3s ease-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        /* Existing styles */
        .payroll-card { margin-bottom: 15px; }
        .payroll-header { background: linear-gradient(45deg, #28a745, #20c997); color: white; }
        .compensation-table { font-size: 0.9rem; }
        .summary-card { background: #f8f9fa; border-left: 4px solid #007bff; }
        .employee-row:hover { background-color: #f8f9fa; }
        .status-badge { font-size: 0.8rem; }
        .debug-info { background: #e9ecef; padding: 10px; margin: 10px 0; border-radius: 4px; font-size: 12px; }
        .action-buttons { margin-top: 15px; }
        .tax-breakdown { background: #fff3cd; border-left: 4px solid #ffc107; }
        .pay-breakdown { background: #e8f5e8; border-left: 4px solid #28a745; }
        .calculating { background: #d1ecf1; border-left: 4px solid #17a2b8; }
        .blink { animation: blink 1s linear infinite; }
        .tax-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 10px; }
        .pay-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 10px; }
        .tax-item { padding: 8px; border-radius: 4px; }
        .pay-item { padding: 8px; border-radius: 4px; }
        .tax-amount { font-weight: bold; }
        .pay-amount { font-weight: bold; color: #28a745; }
        .state-info { background: #e9ecef; padding: 5px 10px; border-radius: 4px; font-size: 0.8rem; }
        .recent-payrolls { background: #f8f9fa; padding: 15px; margin-bottom: 20px; border-radius: 4px; }
        .payroll-link { margin-right: 10px; margin-bottom: 5px; }
        .selected-payroll { background: #e8f5e8 !important; border-left: 4px solid #28a745; }
        .misc-item { background: #e8f5e8 !important; border-left: 3px solid #28a745; }
        @keyframes blink { 50% { opacity: 0.5; } }
    </style>
</head>
<body>
    <!-- Big Flash Messages - FIXED -->
    <?php if(!empty($success_message)): ?>
        <div class="flash-backdrop" id="flashBackdrop"></div>
        <div class="flash-message-container" id="flashMessage">
            <div class="flash-message flash-success">
                <button type="button" class="flash-close" id="flashClose">&times;</button>
                <i class="fa fa-check-circle flash-icon"></i>
                <div><?= htmlspecialchars($success_message) ?></div>
            </div>
        </div>
    <?php endif; ?>
    
    <?php if(!empty($error_message)): ?>
        <div class="flash-backdrop" id="flashBackdrop"></div>
        <div class="flash-message-container" id="flashMessage">
            <div class="flash-message flash-error">
                <button type="button" class="flash-close" id="flashClose">&times;</button>
                <i class="fa fa-exclamation-circle flash-icon"></i>
                <div><?= htmlspecialchars($error_message) ?></div>
            </div>
        </div>
    <?php endif; ?>

    <?php if(!empty($warning_message)): ?>
        <div class="flash-backdrop" id="flashBackdrop"></div>
        <div class="flash-message-container" id="flashMessage">
            <div class="flash-message flash-warning">
                <button type="button" class="flash-close" id="flashClose">&times;</button>
                <i class="fa fa-exclamation-triangle flash-icon"></i>
                <div><?= htmlspecialchars($warning_message) ?></div>
            </div>
        </div>
    <?php endif; ?>

    <div class="container mt-4">
        <h3>Payroll Reports</h3>

        <!-- Debug Info -->
        <div class="debug-info">
            <small>Found <?= count($payrolls) ?> payroll(s) in system | 
            <?php if ($selected_payroll): ?>Viewing: <?= substr($selected_payroll, 0, 8) ?>...<?php else: ?>No payroll selected<?php endif; ?>
            </small>
        </div>

        <!-- Recent Payrolls Quick Links -->
        <?php if (!empty($payrolls)): ?>
        <div class="recent-payrolls">
            <h5>Recent Payrolls (Click to View):</h5>
            <?php foreach ($payrolls as $payroll): 
                $is_selected = $selected_payroll === $payroll['payroll_uuid'];
            ?>
                <a href="gusto_payroll_reports.php?payroll_uuid=<?= $payroll['payroll_uuid'] ?>" 
                   class="btn btn-sm <?= $is_selected ? 'btn-success' : 'btn-outline-primary' ?> payroll-link <?= $is_selected ? 'selected-payroll' : '' ?>">
                    <?= $payroll['week_start'] ?> 
                    (<?= substr($payroll['payroll_uuid'], 0, 8) ?>...)
                    <span class="badge badge-<?= $payroll['status'] == 'failed' ? 'danger' : ($payroll['status'] == 'created' || $payroll['status'] == 'compensation_failed' ? 'warning' : 'success') ?> status-badge">
                        <?= $payroll['status'] ?>
                    </span>
                    <span class="badge badge-info"><?= $payroll['report_count'] ?> reports</span>
                </a>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>

        <!-- Payroll Selection Dropdown -->
        <div class="card mb-4">
            <div class="card-header payroll-header">
                <h5 class="mb-0">Select Payroll Report</h5>
            </div>
            <div class="card-body">
                <?php if (!empty($payrolls)): ?>
                    <form method="post" class="row">
                        <div class="col-md-6">
                            <label class="form-label">Choose Payroll:</label>
                            <select name="payroll_uuid" class="form-control" required>
                                <option value="">Select a payroll...</option>
                                <?php foreach ($payrolls as $payroll): ?>
                                    <option value="<?= $payroll['payroll_uuid'] ?>" 
                                        <?= $selected_payroll == $payroll['payroll_uuid'] ? 'selected' : '' ?>>
                                        Week: <?= $payroll['week_start'] ?> 
                                        | Reports: <?= $payroll['report_count'] ?> 
                                        | Status: <?= $payroll['status'] ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <button type="submit" name="select_payroll" class="btn btn-primary">View Report</button>
                        </div>
                    </form>
                    <?php if ($selected_payroll): ?>
                        <div class="mt-3 p-3 bg-light rounded">
                            <strong>Currently Viewing:</strong> 
                            <?= $payroll_info['week_start'] ?? 'N/A' ?> 
                            (<?= substr($selected_payroll, 0, 8) ?>...)
                            | Reports: <?= count($payroll_reports) ?>
                        </div>
                    <?php endif; ?>
                <?php else: ?>
                    <div class="alert alert-info text-center">
                        <h5>No Payrolls Found</h5>
                        <p class="mb-0"><?= $no_payrolls_message ?? 'Submit a week from the Weekly Reports page first.' ?></p>
                        <a href="gusto_weekly_reports.php" class="btn btn-primary mt-2">Go to Weekly Reports</a>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Payroll Details -->
        <?php if ($selected_payroll): ?>
            <!-- Calculation in Progress -->
            <?php if ($calculation_in_progress): ?>
                <div class="alert alert-info calculating blink">
                    <h5>🔄 Calculation in Progress</h5>
                    <p class="mb-2">Payroll calculation is running. Please wait...</p>
                    <script>
                        setTimeout(function() {
                            window.location.href = 'gusto_payroll_reports.php?payroll_uuid=<?= $selected_payroll ?>';
                        }, 5000);
                    </script>
                </div>
            <?php endif; ?>

            <!-- Payroll Summary -->
            <div class="card mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Payroll Summary</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-3">
                            <div class="summary-card p-3">
                                <h6>Payroll ID</h6>
                                <code class="small"><?= substr($selected_payroll, 0, 20) ?>...</code>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-card p-3">
                                <h6>Week</h6>
                                <strong><?= $payroll_info['week_start'] ?? 'N/A' ?></strong>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-card p-3">
                                <h6>Pay Date</h6>
                                <strong><?= $payroll_info['pay_date'] ?? 'N/A' ?></strong>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-card p-3">
                                <h6>Gusto Status</h6>
                                <?php if (!empty($payroll_details)): ?>
                                    <?php if ($payroll_details['processed']): ?>
                                        <span class="badge badge-success status-badge">PROCESSED</span>
                                    <?php elseif (!empty($payroll_details['calculated_at'])): ?>
                                        <span class="badge badge-info status-badge">CALCULATED</span>
                                    <?php else: ?>
                                        <span class="badge badge-warning status-badge">CREATED</span>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="badge badge-<?= $payroll_info['status'] == 'submitted' ? 'info' : 'warning' ?> status-badge">
                                        <?= strtoupper($payroll_info['status'] ?? 'UNKNOWN') ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Additional Payroll Info -->
                    <?php if (!empty($payroll_details)): ?>
                        <div class="row mt-3">
                            <div class="col-md-4">
                                <strong>Payroll Type:</strong> <?= $payroll_details['off_cycle'] ? 'Off-Cycle' : 'Regular' ?>
                            </div>
                            <div class="col-md-4">
                                <strong>Check Date:</strong> <?= $payroll_details['check_date'] ?? 'N/A' ?>
                            </div>
                            <div class="col-md-4">
                                <strong>Pay Period:</strong> 
                                <?= ($payroll_details['pay_period']['start_date'] ?? 'N/A') . ' to ' . ($payroll_details['pay_period']['end_date'] ?? 'N/A') ?>
                            </div>
                        </div>
                        
                        <!-- Calculate Button for Uncalculated Payrolls -->
                        <?php if (!empty($payroll_details) && empty($payroll_details['calculated_at'])): ?>
                            <div class="action-buttons">
                                <form method="post" class="d-inline">
                                    <input type="hidden" name="payroll_uuid" value="<?= $selected_payroll ?>">
                                    <button type="submit" name="calculate_payroll" class="btn btn-info btn-sm">
                                        🧮 Calculate Payroll
                                    </button>
                                    <small class="text-muted ml-2">
                                        Calculate taxes and deductions in Gusto
                                    </small>
                                </form>
                            </div>
                        <?php endif; ?>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Compensation Data with Detailed Breakdown -->
            <?php if (!empty($employee_compensations)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-success text-white">
                        <h5 class="mb-0">Detailed Compensation Breakdown</h5>
                        <small>Showing solo hours, mileage, bonus, and miscellaneous breakdowns separately</small>
                    </div>
                    <div class="card-body">
                        <?php 
                        $total_gross = 0;
                        $total_net = 0;
                        foreach ($employee_compensations as $comp): 
                            $total_gross += $comp['gross_pay'];
                            $total_net += $comp['net_pay'];
                        ?>
                            <div class="card mb-3">
                                <div class="card-header bg-light">
                                    <h6 class="mb-0">
                                        <?= $comp['employee']['name'] ?? 'Unknown Employee' ?>
                                    </h6>
                                    <?php if (isset($comp['employee_state'])): ?>
                                        <div class="state-info mt-2">
                                            <strong>State:</strong> <?= $comp['employee_state'] ?> 
                                            <?php if (isset($comp['state_tax_rate'])): ?>
                                                | <strong>State Tax Rate:</strong> <?= number_format($comp['state_tax_rate'] * 100, 2) ?>%
                                            <?php endif; ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="card-body">
                                    <!-- Pay Breakdown -->
                                    <div class="pay-breakdown p-3 mb-3">
                                        <h6>Pay Breakdown:</h6>
                                        <div class="pay-grid">
                                            <div class="pay-item bg-light">
                                                <strong>Regular Hours:</strong><br>
                                                <span><?= number_format($comp['pay_breakdown']['regular_hours'] ?? 0, 2) ?> hrs</span><br>
                                                <span class="pay-amount">$<?= number_format($comp['pay_breakdown']['regular_pay'] ?? 0, 2) ?></span>
                                            </div>
                                            <div class="pay-item bg-light">
                                                <strong>Solo Hours (Bonus):</strong><br>
                                                <span><?= number_format($comp['pay_breakdown']['solo_hours'] ?? 0, 2) ?> hrs</span><br>
                                                <span class="pay-amount">$<?= number_format($comp['pay_breakdown']['solo_bonus'] ?? 0, 2) ?></span>
                                            </div>
                                            <div class="pay-item bg-light">
                                                <strong>Mileage:</strong><br>
                                                <span><?= number_format($comp['pay_breakdown']['mileage'] ?? 0, 2) ?> miles</span><br>
                                                <span class="pay-amount">$<?= number_format($comp['pay_breakdown']['mileage_pay'] ?? 0, 2) ?></span>
                                            </div>
                                            <!-- FIXED: Miscellaneous Item -->
                                            <div class="pay-item misc-item">
                                                <strong>Miscellaneous:</strong><br>
                                                <span><?= number_format($comp['pay_breakdown']['miscellaneous'] ?? 0, 2) ?> units</span><br>
                                                <span class="pay-amount">$<?= number_format($comp['pay_breakdown']['miscellaneous_pay'] ?? 0, 2) ?></span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row mb-3">
                                        <div class="col-md-3">
                                            <strong>Gross Pay:</strong><br>
                                            <span class="h5">$<?= number_format($comp['gross_pay'] ?? 0, 2) ?></span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Net Pay:</strong><br>
                                            <span class="h5 text-success">$<?= number_format($comp['net_pay'] ?? 0, 2) ?></span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Total Hours:</strong><br>
                                            <span class="h5"><?= number_format($comp['hours'] ?? 0, 2) ?></span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Payment Method:</strong><br>
                                            <span><?= $comp['payment_method'] ?? 'Direct Deposit' ?></span>
                                        </div>
                                    </div>
                                    
                                    <div class="tax-breakdown p-3">
                                        <h6>Tax Breakdown:</h6>
                                        <div class="tax-grid">
                                            <div class="tax-item bg-light">
                                                <strong>Federal Income Tax:</strong><br>
                                                <span class="tax-amount">$<?= number_format($comp['employee_taxes']['federal_income_tax'] ?? 0, 2) ?></span>
                                            </div>
                                            <div class="tax-item bg-light">
                                                <strong>Social Security:</strong><br>
                                                <span class="tax-amount">$<?= number_format($comp['employee_taxes']['social_security_tax'] ?? 0, 2) ?></span>
                                            </div>
                                            <div class="tax-item bg-light">
                                                <strong>Medicare:</strong><br>
                                                <span class="tax-amount">$<?= number_format($comp['employee_taxes']['medicare_tax'] ?? 0, 2) ?></span>
                                            </div>
                                            <div class="tax-item bg-light">
                                                <strong>State Income Tax:</strong><br>
                                                <span class="tax-amount">$<?= number_format($comp['employee_taxes']['state_income_tax'] ?? 0, 2) ?></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                        
                        <!-- Total Summary -->
                        <div class="card mt-4">
                            <div class="card-header bg-info text-white">
                                <h6 class="mb-0">Payroll Totals</h6>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <div class="col-md-6">
                                        <strong>Total Gross Pay:</strong>
                                        <span class="h4">$<?= number_format($total_gross, 2) ?></span>
                                    </div>
                                    <div class="col-md-6">
                                        <strong>Total Net Pay:</strong>
                                        <span class="h4 text-success">$<?= number_format($total_net, 2) ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php else: ?>
                <div class="alert alert-warning">
                    <h5>No Compensation Data Available</h5>
                    <p class="mb-0">
                        No weekly reports were found for this payroll. This could be because:
                    </p>
                    <ul class="mb-0 mt-2">
                        <li>The payroll is still processing in Gusto</li>
                        <li>Reports are not yet linked to this payroll UUID</li>
                        <li>There was an error during payroll creation</li>
                    </ul>
                    <p class="mb-0 mt-2">Check the Weekly Reports page to ensure reports are properly submitted.</p>
                </div>
            <?php endif; ?>

        <?php else: ?>
            <div class="alert alert-info text-center">
                <h5>Select a Payroll</h5>
                <p class="mb-0">Choose a payroll from the dropdown above and click "View Report" to see its details and compensation information.</p>
            </div>
        <?php endif; ?>

        <!-- Navigation -->
        <div class="mt-4 text-center">
            <a href="gusto_weekly_reports.php" class="btn btn-primary mr-2">
                ← Back to Weekly Reports
            </a>
            <a href="gusto_employees.php" class="btn btn-secondary">
                ← Back to Employee Mapping
            </a>
        </div>
    </div>

    <!-- Bootstrap JavaScript -->
    <script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Flash message functions - FIXED VERSION
        function closeFlashMessage() {
            console.log('Closing flash message');
            $('.flash-message-container, .flash-backdrop').fadeOut(300, function() {
                $(this).remove();
            });
        }

        // Auto-hide flash messages after 5 seconds
        $(document).ready(function() {
            // Improved flash message handling
            if ($('.flash-message-container').length > 0) {
                console.log('Flash message found, setting up auto-close');
                
                // Auto-close after 5 seconds
                setTimeout(function() {
                    closeFlashMessage();
                }, 5000);

                // Close on close button click - FIXED: Use event delegation
                $(document).on('click', '.flash-close', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('Close button clicked');
                    closeFlashMessage();
                });

                // Close on backdrop click - FIXED: Use event delegation
                $(document).on('click', '.flash-backdrop', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('Backdrop clicked');
                    closeFlashMessage();
                });

                // Close on Escape key
                $(document).on('keydown', function(e) {
                    if (e.key === 'Escape') {
                        console.log('Escape key pressed');
                        closeFlashMessage();
                    }
                });
            }
        });
    </script>
    <script src="js/jquery.nicescroll.js"></script>        
	<script src="js/scripts.js"></script>        <!-- Bootstrap Core JavaScript -->       
	<script src="js/bootstrap.min.js"></script>	
</body>
</html>