<?php
// Turn on error reporting
// ini_set('display_errors', 1);
// ini_set('display_startup_errors', 1);
// error_reporting(E_ALL);

// Include db first
require_once 'db.php';

// Check login
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['SESS_MEMBER_ID'])) {
    header('Location: index.php');
    exit;
}

// Include necessary files
require_once 'gusto_api.php';
$config = require __DIR__ . '/gusto_config.php';

// Include custom fields helper
require_once 'custom_fields_helper.php';

// Store flash messages
$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
$warning_message = $_SESSION['warning'] ?? '';
unset($_SESSION['success'], $_SESSION['error'], $_SESSION['warning']);

// Ensure tables exist
ensureCustomFieldsTables();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_report'])) {
        // Add manual report with custom fields
        $eid = intval($_POST['employee_id']);
        $week = $_POST['week_start'];
        $hours = floatval($_POST['hours_total']);
        $solo = floatval($_POST['solo_hours'] ?? 0);
        $miles = floatval($_POST['miles'] ?? 0);
        $miscellaneous = floatval($_POST['miscellaneous'] ?? 0);
        
        // Get employee data with custom rates
        $employee_check = mysqli_query($con, "
            SELECT id, vName, dAmount, solo_rate, miles_rate, miscellaneous_rate, custom_fields_rates 
            FROM usermst WHERE id = $eid
        ");
        
        if (!$employee_check || mysqli_num_rows($employee_check) == 0) {
            $_SESSION['error'] = "Employee ID $eid not found!";
            header('Location: gusto_weekly_reports.php'); 
            exit;
        }
        
        $employee_data = mysqli_fetch_assoc($employee_check);
        $hrate = floatval($employee_data['dAmount'] ?? 16.00);
        $srate = floatval($employee_data['solo_rate'] ?? 3.00);
        $mrate = floatval($employee_data['miles_rate'] ?? 0.25);
        $misc_rate = floatval($employee_data['miscellaneous_rate'] ?? 1.00);
        
        // Parse custom fields rates JSON
        $custom_rates = json_decode($employee_data['custom_fields_rates'] ?? '{}', true) ?: [];
        
        // Calculate standard pay
        $regular_hours = $hours;
        $regular_pay = $regular_hours * $hrate;
        $solo_bonus_pay = $solo * $srate;
        $mileage_pay = $miles * $mrate;
        $miscellaneous_pay = $miscellaneous * $misc_rate;
        
        // Initialize totals
        $custom_fields_pay = 0;
        $gross_pay = $regular_pay + $solo_bonus_pay + $mileage_pay + $miscellaneous_pay;
        
        // Start transaction
        mysqli_begin_transaction($con);
        
        try {
            // Insert main report
            $stmt = mysqli_prepare($con, "
                INSERT INTO weekly_reports 
                (employee_id, week_start, hours_total, solo_hours, miles, miscellaneous, 
                 hourly_rate, mileage_rate, solo_bonus_rate, regular_pay, solo_bonus_pay, 
                 mileage_pay, miscellaneous_pay, gross_pay) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            mysqli_stmt_bind_param($stmt, "isdddddddddddd", 
                $eid, $week, $hours, $solo, $miles, $miscellaneous, $hrate, $mrate, $srate, 
                $regular_pay, $solo_bonus_pay, $mileage_pay, $miscellaneous_pay, $gross_pay
            );
            
            if (!mysqli_stmt_execute($stmt)) {
                throw new Exception("Failed to insert report: " . mysqli_stmt_error($stmt));
            }
            
            $report_id = mysqli_insert_id($con);
            mysqli_stmt_close($stmt);
            
            // Process custom fields
            if (isset($_POST['custom_field'])) {
                foreach ($_POST['custom_field'] as $field_id => $value) {
                    $field_id = intval($field_id);
                    $value = floatval($value);
                    
                    // Skip if value is 0 and not required
                    if ($value == 0) continue;
                    
                    // Get field details
                    $field_query = mysqli_query($con, "
                        SELECT field_type, calculation_type, rate_field, default_value 
                        FROM custom_payroll_fields 
                        WHERE id = $field_id AND is_active = 1
                    ");
                    
                    if ($field_query && mysqli_num_rows($field_query) > 0) {
                        $field_data = mysqli_fetch_assoc($field_query);
                        
                        // Calculate pay based on calculation type
                        $calculated_pay = calculateCustomFieldPay(
                            $field_data, 
                            $value, 
                            $gross_pay, 
                            $employee_data,
                            $custom_rates
                        );
                        
                        // Add to totals
                        $custom_fields_pay += $calculated_pay;
                        
                        // Insert custom field value
                        $stmt2 = mysqli_prepare($con, "
                            INSERT INTO weekly_report_custom_fields 
                            (weekly_report_id, custom_field_id, field_value, calculated_pay) 
                            VALUES (?, ?, ?, ?)
                            ON DUPLICATE KEY UPDATE 
                            field_value = VALUES(field_value),
                            calculated_pay = VALUES(calculated_pay)
                        ");
                        
                        mysqli_stmt_bind_param($stmt2, "iidd", $report_id, $field_id, $value, $calculated_pay);
                        
                        if (!mysqli_stmt_execute($stmt2)) {
                            throw new Exception("Failed to save custom field: " . mysqli_stmt_error($stmt2));
                        }
                        
                        mysqli_stmt_close($stmt2);
                    }
                }
                
                // Update gross pay with custom fields pay
                $gross_pay += $custom_fields_pay;
                mysqli_query($con, "UPDATE weekly_reports SET gross_pay = $gross_pay WHERE id = $report_id");
            }
            
            // Commit transaction
            mysqli_commit($con);
            
            $_SESSION['success'] = "✅ Report added successfully! " .
                "Employee: {$employee_data['vName']}, " .
                "Week: $week, " .
                "Gross Pay: $" . number_format($gross_pay, 2) . 
                ($custom_fields_pay > 0 ? " (Includes $" . number_format($custom_fields_pay, 2) . " from custom fields)" : "");
                
        } catch (Exception $e) {
            mysqli_rollback($con);
            $_SESSION['error'] = "Error: " . $e->getMessage();
        }
        
        header('Location: gusto_weekly_reports.php'); 
        exit;
    }
    
    // Handle other form submissions...
    // [Keep existing code for setup_auto_payroll, submit_week, etc.]
}

// Get active custom fields for display
$custom_fields = mysqli_query($con, "
    SELECT * FROM custom_payroll_fields 
    WHERE is_active = 1 
    ORDER BY display_order
");

// Get custom fields data structure for JavaScript
$custom_fields_js = [];
while($cf = mysqli_fetch_assoc($custom_fields)) {
    $custom_fields_js[] = [
        'id' => $cf['id'],
        'name' => $cf['field_name'],
        'type' => $cf['field_type'],
        'unit' => $cf['unit_label'],
        'calc_type' => $cf['calculation_type'],
        'rate_field' => $cf['rate_field']
    ];
}
mysqli_data_seek($custom_fields, 0); // Reset pointer

// Get sample employees with custom rates
$sample_employees = mysqli_query($con, "
    SELECT u.id, u.vName, u.dAmount, u.solo_rate, u.miles_rate, u.miscellaneous_rate,
           COUNT(DISTINCT wr.id) as report_count
    FROM usermst u
    LEFT JOIN weekly_reports wr ON wr.employee_id = u.id
    WHERE u.is_active = 1 
    GROUP BY u.id
    ORDER BY u.vName
    LIMIT 15
");


// Get total reports count
$total_reports_result = mysqli_query($con, "SELECT COUNT(*) as total FROM weekly_reports");
$total_reports_data = mysqli_fetch_assoc($total_reports_result);
$total_reports = $total_reports_data['total'];

// Get weeks with unsubmitted reports
$weeks_result = mysqli_query($con, "
    SELECT DISTINCT week_start 
    FROM weekly_reports 
    WHERE gusto_submitted = 0 
    ORDER BY week_start DESC 
    LIMIT 20
");

// Get recent reports with custom fields
$reports_result = mysqli_query($con, "
    SELECT wr.*, u.vName as employee_name, u.gusto_employee_uuid,
           (SELECT SUM(calculated_pay) 
            FROM weekly_report_custom_fields wrcf 
            WHERE wrcf.weekly_report_id = wr.id) as custom_fields_total
    FROM weekly_reports wr 
    JOIN usermst u ON u.id = wr.employee_id 
    WHERE u.is_active = 1 
    ORDER BY wr.week_start DESC, wr.id DESC 
    LIMIT 50
");
?>

<!DOCTYPE html>
<html>
<head>
    <title>MaidToSparkle: Enhanced Weekly Reports</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <link href="css/font-awesome.css" rel="stylesheet">
    <style>
        /* Custom Fields Styles */
        .custom-field-group {
            background: #f8f9fa;
            border: 1px dashed #6c757d;
            border-radius: 6px;
            padding: 15px;
            margin-bottom: 15px;
        }
        .custom-field-input {
            border-left: 3px solid #17a2b8;
            background-color: #f8ffff;
        }
        .custom-field-value {
            background-color: #e8f5e8;
            color: #155724;
            padding: 2px 6px;
            border-radius: 3px;
            font-weight: bold;
        }
        .field-preview {
            background: #fff3cd;
            border-left: 3px solid #ffc107;
            padding: 10px;
            margin-top: 5px;
            border-radius: 4px;
            font-size: 0.9rem;
        }
        .dynamic-fields-container {
            max-height: 300px;
            overflow-y: auto;
            padding-right: 10px;
        }
        .field-type-icon {
            width: 24px;
            height: 24px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
            margin-right: 8px;
            font-size: 0.8rem;
        }
        .hours-icon { background: #17a2b8; color: white; }
        .dollars-icon { background: #28a745; color: white; }
        .miles-icon { background: #007bff; color: white; }
        .units-icon { background: #6f42c1; color: white; }
        .percentage-icon { background: #fd7e14; color: white; }
        .employee-rate-badge {
            background: #e9ecef;
            color: #495057;
            font-size: 0.8rem;
            padding: 2px 6px;
            border-radius: 3px;
            margin-left: 5px;
        }
    </style>
</head>
<body>
    <div class="container mt-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Enhanced Weekly Reports</h3>
                <p class="text-muted mb-0">Add reports with unlimited custom fields</p>
            </div>
            <div>
                <a href="custom_payroll_fields.php" class="btn btn-outline-primary">
                    <i class="fa fa-cog"></i> Manage Fields
                </a>
            </div>
        </div>
        
        <!-- Flash Messages -->
        <?php if($success_message): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <?= $success_message ?>
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        <?php endif; ?>
        
        <?php if($error_message): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <?= $error_message ?>
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        <?php endif; ?>
        
        <!-- Quick Employee Selection -->
        <?php if ($sample_employees && mysqli_num_rows($sample_employees) > 0): ?>
        <!-- <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="fa fa-users"></i> Quick Employee Selection</h5>
            </div>
            <div class="card-body">
                <p class="mb-3">Click an employee to auto-fill the form:</p>
                <div class="row">
                    <?php while ($emp = mysqli_fetch_assoc($sample_employees)): 
                        $custom_rates = json_decode($emp['custom_fields_rates'] ?? '{}', true);
                        $has_custom_rates = !empty($custom_rates);
                    ?>
                        <div class="col-md-3 mb-3">
                            <button class="btn btn-outline-primary btn-block text-left set-employee" 
                                    data-id="<?= $emp['id'] ?>"
                                    data-name="<?= htmlspecialchars($emp['vName']) ?>"
                                    data-hourly="<?= $emp['dAmount'] ?>"
                                    data-misc="<?= $emp['miscellaneous_rate'] ?>">
                                <div class="d-flex justify-content-between">
                                    <strong><?= htmlspecialchars($emp['vName']) ?></strong>
                                    <span class="badge badge-info"><?= $emp['report_count'] ?></span>
                                </div>
                                <small>ID: <?= $emp['id'] ?> | $<?= number_format($emp['dAmount'], 2) ?>/hr</small>
                                <?php if($has_custom_rates): ?>
                                    <div class="mt-1">
                                        <small class="text-success">
                                            <i class="fa fa-star"></i> Custom rates available
                                        </small>
                                    </div>
                                <?php endif; ?>
                            </button>
                        </div>
                    <?php endwhile; ?>
                </div>
            </div>
        </div> -->
        <?php endif; ?>
        
        <!-- Add Manual Report with Custom Fields -->
        <div class="card mb-4">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0"><i class="fa fa-plus-circle"></i> Add Report with Custom Fields</h5>
            </div>
            <div class="card-body">
                <form method="post" id="reportForm" class="row g-3">
                    <div class="col-md-12">
                        <div class="alert alert-info">
                            <i class="fa fa-info-circle"></i> 
                            <strong>Note:</strong> Custom fields will be calculated based on their calculation type and employee rates.
                        </div>
                    </div>
                    
                    <!-- Basic Information -->
                    <div class="col-md-3">
                        <label class="form-label">Week Start *</label>
                        <input type="date" name="week_start" class="form-control" 
                               value="<?= date('Y-m-d', strtotime('monday this week')) ?>" required>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Employee ID *</label>
                        <div class="input-group">
                            <input type="number" step="1" name="employee_id" id="employee_id" 
                                   class="form-control" placeholder="Employee ID" required>
                            <div class="input-group-append">
                                <button type="button" class="btn btn-outline-secondary" 
                                        onclick="loadEmployeeRates()">
                                    <i class="fa fa-refresh"></i> Load Rates
                                </button>
                            </div>
                        </div>
                        <small class="text-muted" id="employeeInfo"></small>
                    </div>
                    
                    <!-- Standard Fields -->
                    <div class="col-md-1">
                        <label class="form-label">Hours *</label>
                        <input type="number" step="0.1" name="hours_total" 
                               class="form-control" placeholder="Hours" required>
                    </div>
                    <div class="col-md-1">
                        <label class="form-label">Solo</label>
                        <input type="number" step="0.1" name="solo_hours" 
                               class="form-control" placeholder="Solo">
                    </div>
                    <div class="col-md-1">
                        <label class="form-label">Miles</label>
                        <input type="number" step="0.1" name="miles" 
                               class="form-control" placeholder="Miles">
                    </div>
                    <div class="col-md-1">
                        <label class="form-label">Misc</label>
                        <input type="number" step="0.1" name="miscellaneous" 
                               class="form-control" placeholder="Misc" value="0">
                    </div>
                    
                    <!-- Custom Fields Section -->
                    <div class="col-md-12 mt-3">
                        <div class="custom-field-group">
                            <h6><i class="fa fa-list-alt"></i> Custom Fields</h6>
                            <p class="text-muted small mb-3">
                                Fill in custom field values. Pay will be calculated automatically.
                            </p>
                            
                            <div class="dynamic-fields-container" id="customFieldsContainer">
                                <?php if(mysqli_num_rows($custom_fields) > 0): ?>
                                    <?php while($cf = mysqli_fetch_assoc($custom_fields)): 
                                        $icon_class = '';
                                        switch($cf['field_type']) {
                                            case 'hours': $icon_class = 'hours-icon'; break;
                                            case 'dollars': $icon_class = 'dollars-icon'; break;
                                            case 'miles': $icon_class = 'miles-icon'; break;
                                            case 'units': $icon_class = 'units-icon'; break;
                                            case 'percentage': $icon_class = 'percentage-icon'; break;
                                            default: $icon_class = 'hours-icon';
                                        }
                                    ?>
                                        <div class="form-group mb-3 custom-field-input" 
                                             data-field-id="<?= $cf['id'] ?>"
                                             data-calc-type="<?= $cf['calculation_type'] ?>"
                                             data-rate-field="<?= $cf['rate_field'] ?>">
                                            <label class="small font-weight-bold">
                                                <span class="field-type-icon <?= $icon_class ?>">
                                                    <?= strtoupper(substr($cf['field_type'], 0, 1)) ?>
                                                </span>
                                                <?= htmlspecialchars($cf['field_name']) ?>
                                                <?php if($cf['is_required']): ?>
                                                    <span class="text-danger">*</span>
                                                <?php endif; ?>
                                                <?php if($cf['unit_label']): ?>
                                                    <span class="text-muted">(<?= $cf['unit_label'] ?>)</span>
                                                <?php endif; ?>
                                                <small class="text-info ml-2">
                                                    [<?= str_replace('_', ' ', $cf['calculation_type']) ?>]
                                                </small>
                                            </label>
                                            
                                            <div class="input-group">
                                                <?php if($cf['field_type'] == 'text'): ?>
                                                    <input type="text" 
                                                           name="custom_field[<?= $cf['id'] ?>]" 
                                                           class="form-control form-control-sm custom-field-value" 
                                                           placeholder="Enter text"
                                                           data-field-type="text"
                                                           <?= $cf['is_required'] ? 'required' : '' ?>>
                                                <?php else: ?>
                                                    <input type="number" step="0.01" 
                                                           name="custom_field[<?= $cf['id'] ?>]" 
                                                           class="form-control form-control-sm custom-field-value" 
                                                           placeholder="0.00"
                                                           value="<?= $cf['default_value'] ?>"
                                                           data-field-type="<?= $cf['field_type'] ?>"
                                                           <?= $cf['is_required'] ? 'required' : '' ?>
                                                           onchange="updateFieldPreview(this)">
                                                <?php endif; ?>
                                                <div class="input-group-append">
                                                    <span class="input-group-text bg-light">
                                                        <small><?= $cf['unit_label'] ?: 'value' ?></small>
                                                    </span>
                                                </div>
                                            </div>
                                            
                                            <div class="field-preview" id="preview_<?= $cf['id'] ?>">
                                                <small class="text-muted">Enter value to see calculation preview</small>
                                            </div>
                                        </div>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <div class="alert alert-warning">
                                        No custom fields defined. 
                                        <a href="custom_payroll_fields.php">Create some custom fields first.</a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Summary Preview -->
                    <div class="col-md-12">
                        <div class="card bg-light">
                            <div class="card-body">
                                <h6>Pay Summary Preview</h6>
                                <div class="row" id="paySummary">
                                    <div class="col-md-3">
                                        <small class="text-muted">Regular Pay:</small>
                                        <div id="regularPay">$0.00</div>
                                    </div>
                                    <div class="col-md-3">
                                        <small class="text-muted">Solo Bonus:</small>
                                        <div id="soloPay">$0.00</div>
                                    </div>
                                    <div class="col-md-3">
                                        <small class="text-muted">Mileage:</small>
                                        <div id="mileagePay">$0.00</div>
                                    </div>
                                    <div class="col-md-3">
                                        <small class="text-muted">Custom Fields:</small>
                                        <div id="customFieldsPay">$0.00</div>
                                    </div>
                                    <div class="col-md-12 mt-2">
                                        <hr>
                                        <div class="d-flex justify-content-between">
                                            <strong>Total Gross Pay:</strong>
                                            <strong id="totalGrossPay">$0.00</strong>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Submit Button -->
                    <div class="col-md-12 mt-3">
                        <button name="add_report" class="btn btn-success btn-lg">
                            <i class="fa fa-check-circle"></i> Add Report with All Fields
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="calculatePreview()">
                            <i class="fa fa-calculator"></i> Calculate Preview
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Reports Table with Custom Fields -->
        <div class="card">
            <div class="card-header bg-light d-flex justify-content-between align-items-center">
                <h5 class="mb-0">All Reports with Custom Fields (<?= $total_reports ?> total)</h5>
                <div>
                    <button class="btn btn-sm btn-outline-info" onclick="toggleCustomFields()">
                        <i class="fa fa-eye"></i> Toggle Custom Fields
                    </button>
                </div>
            </div>
            <div class="card-body">
                <?php if ($total_reports > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-sm table-striped" id="reportsTable">
                            <thead>
                                <tr>
                                    <th>Week</th>
                                    <th>Employee</th>
                                    <th>Hours</th>
                                    <th>Solo</th>
                                    <th>Miles</th>
                                    <th>Misc</th>
                                    <th class="custom-field-column">Custom Fields</th>
                                    <th>Regular Pay</th>
                                    <th>Solo Bonus</th>
                                    <th>Mileage</th>
                                    <th>Custom Pay</th>
                                    <th>Gross Pay</th>
                                    <th>Submitted</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while($r = mysqli_fetch_assoc($reports_result)): 
                                    $row_class = $r['gusto_submitted'] ? 'table-success' : '';
                                    $custom_fields_total = $r['custom_fields_total'] ?? 0;
                                ?>
                                    <tr class="<?= $row_class ?>">
                                        <td><?= $r['week_start'] ?></td>
                                        <td>
                                            <?= $r['employee_name'] ?>
                                            <small class="d-block text-muted">ID: <?= $r['employee_id'] ?></small>
                                        </td>
                                        <td><?= number_format($r['hours_total'], 2) ?></td>
                                        <td><?= number_format($r['solo_hours'], 2) ?></td>
                                        <td><?= number_format($r['miles'], 2) ?></td>
                                        <td><?= number_format($r['miscellaneous'], 2) ?></td>
                                        <td class="custom-field-column">
                                            <?php 
                                            // Get custom fields for this report
                                            $cf_query = mysqli_query($con, "
                                                SELECT cpf.field_name, cpf.unit_label, wrf.field_value, wrf.calculated_pay
                                                FROM weekly_report_custom_fields wrf
                                                JOIN custom_payroll_fields cpf ON cpf.id = wrf.custom_field_id
                                                WHERE wrf.weekly_report_id = {$r['id']}
                                            ");
                                            
                                            if(mysqli_num_rows($cf_query) > 0): 
                                                while($cf = mysqli_fetch_assoc($cf_query)):
                                            ?>
                                                <div class="small mb-1">
                                                    <span class="badge badge-light"><?= $cf['field_name'] ?></span>
                                                    <span class="text-muted"><?= $cf['field_value'] ?> <?= $cf['unit_label'] ?></span>
                                                    <span class="badge badge-info">$<?= number_format($cf['calculated_pay'], 2) ?></span>
                                                </div>
                                            <?php 
                                                endwhile;
                                            else:
                                                echo '<span class="text-muted">-</span>';
                                            endif; 
                                            ?>
                                        </td>
                                        <td>$<?= number_format($r['regular_pay'], 2) ?></td>
                                        <td>$<?= number_format($r['solo_bonus_pay'], 2) ?></td>
                                        <td>$<?= number_format($r['mileage_pay'], 2) ?></td>
                                        <td>
                                            <?php if($custom_fields_total > 0): ?>
                                                <span class="badge badge-success">$<?= number_format($custom_fields_total, 2) ?></span>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <strong>$<?= number_format($r['gross_pay'], 2) ?></strong>
                                            <?php if($custom_fields_total > 0): ?>
                                                <small class="d-block text-success">
                                                    +$<?= number_format($custom_fields_total, 2) ?> custom
                                                </small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if($r['gusto_submitted']): ?>
                                                <span class="badge badge-success">Yes</span>
                                            <?php else: ?>
                                                <span class="badge badge-warning">No</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="view_report_details.php?id=<?= $r['id'] ?>" 
                                               class="btn btn-sm btn-outline-info">
                                                <i class="fa fa-eye"></i> Details
                                            </a>
                                        </td>
                                    </tr>
                                <?php endwhile; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="alert alert-info text-center">
                        <h5>No Reports Yet</h5>
                        <p class="mb-0">Add your first report using the form above.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="js/jquery-1.12.4.js"></script>
    <script src="js/bootstrap.min.js"></script>
    <script>
        // Custom fields data
        const customFields = <?= json_encode($custom_fields_js) ?>;
        
        // Employee data cache
        let employeeRates = {
            hourly: 16.00,
            solo: 3.00,
            miles: 0.25,
            misc: 1.00,
            custom: {}
        };
        
        // Initialize
        $(document).ready(function() {
            // Set up employee selection
            $('.set-employee').click(function(e) {
                e.preventDefault();
                const employeeId = $(this).data('id');
                const employeeName = $(this).data('name');
                const hourlyRate = $(this).data('hourly');
                const miscRate = $(this).data('misc');
                
                $('#employee_id').val(employeeId);
                $('#employeeInfo').html(`<i class="fa fa-user"></i> ${employeeName} | $${hourlyRate}/hr`);
                
                // Load employee rates
                loadEmployeeRates();
            });
            
            // Calculate preview when inputs change
            $('input[name="hours_total"], input[name="solo_hours"], input[name="miles"], input[name="miscellaneous"]').on('input', calculatePreview);
            
            // Initialize preview
            calculatePreview();
        });
        
        function loadEmployeeRates() {
            const employeeId = $('#employee_id').val();
            if (!employeeId) {
                alert('Please enter an Employee ID first');
                return;
            }
            
            $.ajax({
                url: 'get_employee_rates.php',
                type: 'GET',
                data: { employee_id: employeeId },
                success: function(response) {
                    if (response.success) {
                        employeeRates = response.rates;
                        $('#employeeInfo').html(`
                            <i class="fa fa-check text-success"></i> 
                            ${response.name} | 
                            Hourly: $${response.rates.hourly} | 
                            Solo: $${response.rates.solo} | 
                            Miles: $${response.rates.miles} | 
                            Misc: $${response.rates.misc}
                        `);
                        
                        // Update all field previews
                        $('.custom-field-value').each(function() {
                            updateFieldPreview($(this)[0]);
                        });
                        
                        calculatePreview();
                    } else {
                        alert('Employee not found or error loading rates');
                    }
                },
                error: function() {
                    alert('Error loading employee rates');
                }
            });
        }
        
        function updateFieldPreview(input) {
            const fieldId = $(input).closest('.custom-field-input').data('field-id');
            const value = parseFloat($(input).val()) || 0;
            const field = customFields.find(f => f.id == fieldId);
            
            if (!field) return;
            
            let previewText = '';
            let calculatedAmount = 0;
            
            switch(field.calc_type) {
                case 'multiply_rate':
                    const rateField = field.rate_field;
                    const rate = employeeRates[rateField] || employeeRates[rateField.replace('_rate', '')] || 0;
                    calculatedAmount = value * rate;
                    previewText = `${value} × $${rate} = $${calculatedAmount.toFixed(2)}`;
                    break;
                    
                case 'fixed_amount':
                    calculatedAmount = value;
                    previewText = `Fixed amount: $${calculatedAmount.toFixed(2)}`;
                    break;
                    
                case 'percentage_of_gross':
                    // Will be calculated in preview
                    previewText = `${value}% of gross pay`;
                    break;
                    
                default:
                    previewText = `Value: ${value}`;
            }
            
            $(`#preview_${fieldId}`).html(`
                <strong>${previewText}</strong>
                ${calculatedAmount > 0 ? `<span class="float-right text-success">$${calculatedAmount.toFixed(2)}</span>` : ''}
            `);
        }
        
        function calculatePreview() {
            // Get standard values
            const hours = parseFloat($('input[name="hours_total"]').val()) || 0;
            const solo = parseFloat($('input[name="solo_hours"]').val()) || 0;
            const miles = parseFloat($('input[name="miles"]').val()) || 0;
            const misc = parseFloat($('input[name="miscellaneous"]').val()) || 0;
            
            // Calculate standard pay
            const regularPay = hours * employeeRates.hourly;
            const soloPay = solo * employeeRates.solo;
            const mileagePay = miles * employeeRates.miles;
            const miscPay = misc * employeeRates.misc;
            
            // Calculate custom fields pay
            let customFieldsPay = 0;
            $('.custom-field-value').each(function() {
                const value = parseFloat($(this).val()) || 0;
                const fieldDiv = $(this).closest('.custom-field-input');
                const fieldId = fieldDiv.data('field-id');
                const calcType = fieldDiv.data('calc-type');
                const rateField = fieldDiv.data('rate-field');
                
                if (value > 0) {
                    switch(calcType) {
                        case 'multiply_rate':
                            const rate = employeeRates[rateField] || employeeRates[rateField.replace('_rate', '')] || 0;
                            customFieldsPay += value * rate;
                            break;
                        case 'fixed_amount':
                            customFieldsPay += value;
                            break;
                        case 'percentage_of_gross':
                            // Will be calculated after standard pay
                            break;
                    }
                }
            });
            
            // Calculate total before percentage fields
            let totalBeforePercentage = regularPay + soloPay + mileagePay + miscPay + customFieldsPay;
            
            // Add percentage fields (percentage of the total before percentage)
            $('.custom-field-value').each(function() {
                const value = parseFloat($(this).val()) || 0;
                const fieldDiv = $(this).closest('.custom-field-input');
                const calcType = fieldDiv.data('calc-type');
                
                if (value > 0 && calcType === 'percentage_of_gross') {
                    customFieldsPay += (value / 100) * totalBeforePercentage;
                }
            });
            
            // Calculate final total
            const totalGross = regularPay + soloPay + mileagePay + miscPay + customFieldsPay;
            
            // Update display
            $('#regularPay').html(`$${regularPay.toFixed(2)}`);
            $('#soloPay').html(`$${soloPay.toFixed(2)}`);
            $('#mileagePay').html(`$${mileagePay.toFixed(2)}`);
            $('#customFieldsPay').html(`$${customFieldsPay.toFixed(2)}`);
            $('#totalGrossPay').html(`$${totalGross.toFixed(2)}`);
        }
        
        function toggleCustomFields() {
            $('.custom-field-column').toggle();
        }
    </script>
</body>
</html>