<?php


session_start();
if (!isset($_SESSION['SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';

// Include integration files
require_once 'custom_fields_helper.php';
require_once 'quickbooks_integration.php';
require_once 'guru_integration.php';

$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
$warning_message = $_SESSION['warning'] ?? '';
unset($_SESSION['success'], $_SESSION['error'], $_SESSION['warning']);

// Get system stats
$stats = [
    'custom_fields' => [
        'total' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as total FROM custom_payroll_fields"
        ))['total'],
        'active' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as active FROM custom_payroll_fields WHERE is_active = 1"
        ))['active'],
        'usage' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(DISTINCT weekly_report_id) as usage FROM weekly_report_custom_fields"
        ))['usage']
    ],
    
    'gusto' => [
        'submitted' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as submitted FROM weekly_reports WHERE gusto_submitted = 1"
        ))['submitted'],
        'pending' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as pending FROM weekly_reports WHERE gusto_submitted = 0"
        ))['pending'],
        'payrolls' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as payrolls FROM payrolls"
        ))['payrolls']
    ],
    
    'quickbooks' => [
        'synced' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as synced FROM weekly_reports WHERE quickbooks_synced = 1"
        ))['synced'],
        'pending' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as pending FROM weekly_reports WHERE gusto_submitted = 1 AND quickbooks_synced = 0"
        ))['pending'],
        'vendors' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(DISTINCT quickbooks_vendor_id) as vendors FROM usermst WHERE quickbooks_vendor_id IS NOT NULL"
        ))['vendors']
    ],
    
    'guru' => [
        'synced' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as synced FROM weekly_reports WHERE guru_synced = 1"
        ))['synced'],
        'pending' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(*) as pending FROM weekly_reports WHERE gusto_submitted = 1 AND guru_synced = 0"
        ))['pending'],
        'users' => mysqli_fetch_assoc(mysqli_query($con, 
            "SELECT COUNT(DISTINCT guru_user_id) as users FROM usermst WHERE guru_user_id IS NOT NULL"
        ))['users']
    ]
];

// Get connection status
$qb_token = getQuickBooksAccessToken();
$guru_token = getGuruAccessToken();

// Get recent sync logs
$recent_logs = mysqli_query($con, "
    SELECT * FROM integration_sync_logs 
    ORDER BY created_at DESC 
    LIMIT 10
");

// Get system health
$system_health = [
    'database' => checkDatabaseHealth(),
    'gusto_api' => checkGustoHealth(),
    'quickbooks_api' => $qb_token ? 'Connected' : 'Disconnected',
    'guru_api' => $guru_token ? 'Connected' : 'Disconnected',
    'custom_fields' => $stats['custom_fields']['total'] > 0 ? 'Active' : 'Not Configured'
];

function checkDatabaseHealth() {
    global $con;
    $result = mysqli_query($con, "SELECT 1 as health_check");
    return $result ? 'Healthy' : 'Issues';
}

function checkGustoHealth() {
    global $config;
    require_once 'gusto_api.php';
    $token = get_gusto_access_token();
    return $token ? 'Connected' : 'Disconnected';
}

// Handle sync all action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['sync_all'])) {
    // This would trigger sync across all systems
    $_SESSION['success'] = "Sync initiated across all systems. Check logs for details.";
    header('Location: integration_dashboard.php');
    exit;
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>MaidToSparkle: Integration Dashboard</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <link href="css/font-awesome.css" rel="stylesheet">
    <style>
        .dashboard-card {
            border-radius: 10px;
            border: none;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            transition: transform 0.2s;
        }
        .dashboard-card:hover {
            transform: translateY(-2px);
        }
        .card-header-custom {
            border-radius: 10px 10px 0 0 !important;
            color: white;
            padding: 15px 20px;
        }
        .card-gusto { border-top: 4px solid #28a745; }
        .card-qb { border-top: 4px solid #17a2b8; }
        .card-guru { border-top: 4px solid #6f42c1; }
        .card-fields { border-top: 4px solid #ffc107; }
        .health-status {
            display: inline-block;
            width: 10px;
            height: 10px;
            border-radius: 50%;
            margin-right: 5px;
        }
        .health-good { background-color: #28a745; }
        .health-warning { background-color: #ffc107; }
        .health-error { background-color: #dc3545; }
        .sync-button {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
        }
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            line-height: 1;
        }
        .stat-label {
            font-size: 0.85rem;
            color: #6c757d;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        .log-row {
            border-left: 3px solid #dee2e6;
            padding-left: 10px;
            margin-bottom: 8px;
        }
        .log-success { border-left-color: #28a745; }
        .log-error { border-left-color: #dc3545; }
        .log-warning { border-left-color: #ffc107; }
    </style>
</head>
<body>
    <div class="container-fluid mt-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h3 class="mb-1">Integration Dashboard</h3>
                <p class="text-muted mb-0">Overview of all payroll system integrations</p>
            </div>
            <div>
                <a href="gusto_weekly_reports_enhanced.php" class="btn btn-outline-primary">
                    <i class="fa fa-plus-circle"></i> Create Report
                </a>
            </div>
        </div>
        
        <?php if($success_message): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <?= $success_message ?>
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        <?php endif; ?>
        
        <?php if($error_message): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <?= $error_message ?>
                <button type="button" class="close" data-dismiss="alert">&times;</button>
            </div>
        <?php endif; ?>
        
        <!-- System Health Status -->
        <div class="card dashboard-card mb-4">
            <div class="card-header card-header-custom bg-dark">
                <h5 class="mb-0"><i class="fa fa-heartbeat"></i> System Health Status</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <?php foreach($system_health as $system => $status): 
                        $health_class = $status === 'Healthy' || $status === 'Connected' || $status === 'Active' ? 'health-good' : 
                                      ($status === 'Issues' || $status === 'Disconnected' ? 'health-error' : 'health-warning');
                    ?>
                        <div class="col-md-2 col-6 mb-3">
                            <div class="text-center">
                                <span class="health-status <?= $health_class ?>"></span>
                                <div class="small text-muted"><?= ucfirst(str_replace('_', ' ', $system)) ?></div>
                                <div class="font-weight-bold"><?= $status ?></div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Stats Cards -->
        <div class="row mb-4">
            <!-- Gusto Stats -->
            <div class="col-md-3">
                <div class="card dashboard-card card-gusto">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <div class="stat-number text-success"><?= $stats['gusto']['submitted'] ?></div>
                                <div class="stat-label">Gusto Submissions</div>
                            </div>
                            <div class="text-success">
                                <i class="fa fa-briefcase fa-2x"></i>
                            </div>
                        </div>
                        <div class="mt-3">
                            <small class="text-muted">
                                <i class="fa fa-clock"></i> <?= $stats['gusto']['pending'] ?> pending
                                | <i class="fa fa-file"></i> <?= $stats['gusto']['payrolls'] ?> payrolls
                            </small>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="gusto_payroll_reports.php" class="btn btn-sm btn-outline-success btn-block">
                            View Reports
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- QuickBooks Stats -->
            <div class="col-md-3">
                <div class="card dashboard-card card-qb">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <div class="stat-number text-info"><?= $stats['quickbooks']['synced'] ?></div>
                                <div class="stat-label">QuickBooks Syncs</div>
                            </div>
                            <div class="text-info">
                                <i class="fa fa-database fa-2x"></i>
                            </div>
                        </div>
                        <div class="mt-3">
                            <small class="text-muted">
                                <i class="fa fa-clock"></i> <?= $stats['quickbooks']['pending'] ?> pending
                                | <i class="fa fa-users"></i> <?= $stats['quickbooks']['vendors'] ?> vendors
                            </small>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="quickbooks_integration.php" class="btn btn-sm btn-outline-info btn-block">
                            Manage QB
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Guru Stats -->
            <div class="col-md-3">
                <div class="card dashboard-card card-guru">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <div class="stat-number text-purple"><?= $stats['guru']['synced'] ?></div>
                                <div class="stat-label">Guru Syncs</div>
                            </div>
                            <div class="text-purple">
                                <i class="fa fa-users fa-2x"></i>
                            </div>
                        </div>
                        <div class="mt-3">
                            <small class="text-muted">
                                <i class="fa fa-clock"></i> <?= $stats['guru']['pending'] ?> pending
                                | <i class="fa fa-user"></i> <?= $stats['guru']['users'] ?> users
                            </small>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="guru_integration.php" class="btn btn-sm btn-outline-purple btn-block">
                            Manage Guru
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Custom Fields Stats -->
            <div class="col-md-3">
                <div class="card dashboard-card card-fields">
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <div class="stat-number text-warning"><?= $stats['custom_fields']['active'] ?></div>
                                <div class="stat-label">Custom Fields</div>
                            </div>
                            <div class="text-warning">
                                <i class="fa fa-list-alt fa-2x"></i>
                            </div>
                        </div>
                        <div class="mt-3">
                            <small class="text-muted">
                                <i class="fa fa-check"></i> <?= $stats['custom_fields']['total'] ?> total
                                | <i class="fa fa-chart-bar"></i> <?= $stats['custom_fields']['usage'] ?> used
                            </small>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="custom_payroll_fields.php" class="btn btn-sm btn-outline-warning btn-block">
                            Manage Fields
                        </a>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Quick Actions -->
        <div class="card dashboard-card mb-4">
            <div class="card-header card-header-custom bg-primary">
                <h5 class="mb-0"><i class="fa fa-bolt"></i> Quick Actions</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <a href="gusto_weekly_reports_enhanced.php" class="btn btn-outline-success btn-block">
                            <i class="fa fa-plus"></i> Create Report
                        </a>
                    </div>
                    <div class="col-md-3 mb-3">
                        <a href="sync_payroll_to_qb.php" class="btn btn-outline-info btn-block">
                            <i class="fa fa-sync"></i> Sync to QuickBooks
                        </a>
                    </div>
                    <div class="col-md-3 mb-3">
                        <a href="sync_timesheets_to_guru.php" class="btn btn-outline-purple btn-block">
                            <i class="fa fa-share"></i> Sync to Guru
                        </a>
                    </div>
                    <div class="col-md-3 mb-3">
                        <form method="post" class="mb-0">
                            <button type="submit" name="sync_all" class="btn btn-outline-warning btn-block">
                                <i class="fa fa-sync-alt"></i> Sync All Systems
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Recent Activity Logs -->
        <div class="row">
            <div class="col-md-8">
                <div class="card dashboard-card">
                    <div class="card-header card-header-custom bg-dark">
                        <h5 class="mb-0"><i class="fa fa-history"></i> Recent Activity Logs</h5>
                    </div>
                    <div class="card-body">
                        <div class="activity-feed">
                            <?php if(mysqli_num_rows($recent_logs) > 0): ?>
                                <?php while($log = mysqli_fetch_assoc($recent_logs)): 
                                    $log_class = 'log-' . $log['status'];
                                ?>
                                    <div class="log-row <?= $log_class ?>">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <strong>
                                                    <span class="badge badge-<?= $log['status'] == 'success' ? 'success' : ($log['status'] == 'error' ? 'danger' : 'warning') ?>">
                                                        <?= $log['integration_type'] ?>
                                                    </span>
                                                    <?= $log['action_type'] ?>
                                                </strong>
                                                <?php if($log['entity_type']): ?>
                                                    <small class="text-muted">(<?= $log['entity_type'] ?> <?= $log['entity_id'] ?>)</small>
                                                <?php endif; ?>
                                            </div>
                                            <div class="text-muted small">
                                                <?= date('H:i', strtotime($log['created_at'])) ?>
                                            </div>
                                        </div>
                                        <div class="small mt-1">
                                            <?= substr($log['message'], 0, 100) ?>...
                                        </div>
                                    </div>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <div class="text-center text-muted py-3">
                                    <i class="fa fa-info-circle fa-2x mb-2"></i>
                                    <p>No activity logs yet</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="integration_reports.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fa fa-file-alt"></i> View All Logs
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- System Info -->
            <div class="col-md-4">
                <div class="card dashboard-card">
                    <div class="card-header card-header-custom bg-info">
                        <h5 class="mb-0"><i class="fa fa-info-circle"></i> System Information</h5>
                    </div>
                    <div class="card-body">
                        <ul class="list-unstyled mb-0">
                            <li class="mb-2">
                                <i class="fa fa-server text-muted mr-2"></i>
                                <strong>PHP Version:</strong> <?= phpversion() ?>
                            </li>
                            <li class="mb-2">
                                <i class="fa fa-database text-muted mr-2"></i>
                                <strong>MySQL Version:</strong> 
                                <?php
                                $mysql_version = mysqli_get_server_info($con);
                                echo substr($mysql_version, 0, 20);
                                ?>
                            </li>
                            <li class="mb-2">
                                <i class="fa fa-clock text-muted mr-2"></i>
                                <strong>Last Backup:</strong> 
                                <?php
                                $backup_file = 'backup/last_backup.txt';
                                if (file_exists($backup_file)) {
                                    echo date('Y-m-d H:i', filemtime($backup_file));
                                } else {
                                    echo 'Never';
                                }
                                ?>
                            </li>
                            <li class="mb-2">
                                <i class="fa fa-hdd text-muted mr-2"></i>
                                <strong>Disk Usage:</strong> 
                                <?php
                                $free = disk_free_space(".");
                                $total = disk_total_space(".");
                                $used = $total - $free;
                                $percent_used = round(($used / $total) * 100, 1);
                                echo $percent_used . '%';
                                ?>
                            </li>
                            <li>
                                <i class="fa fa-calendar text-muted mr-2"></i>
                                <strong>System Time:</strong> <?= date('Y-m-d H:i:s') ?>
                            </li>
                        </ul>
                    </div>
                    <div class="card-footer bg-transparent">
                        <a href="api_monitor.php" class="btn btn-sm btn-outline-info">
                            <i class="fa fa-chart-line"></i> System Monitor
                        </a>
                    </div>
                </div>
                
                <!-- Quick Links -->
                <div class="card dashboard-card mt-4">
                    <div class="card-header card-header-custom bg-warning">
                        <h5 class="mb-0"><i class="fa fa-link"></i> Quick Links</h5>
                    </div>
                    <div class="card-body">
                        <div class="list-group list-group-flush">
                            <a href="custom_payroll_fields.php" class="list-group-item list-group-item-action">
                                <i class="fa fa-list-alt mr-2"></i> Custom Fields
                            </a>
                            <a href="quickbooks_integration.php" class="list-group-item list-group-item-action">
                                <i class="fa fa-database mr-2"></i> QuickBooks
                            </a>
                            <a href="guru_integration.php" class="list-group-item list-group-item-action">
                                <i class="fa fa-users mr-2"></i> Guru
                            </a>
                            <a href="view_logs.php" class="list-group-item list-group-item-action">
                                <i class="fa fa-file-alt mr-2"></i> System Logs
                            </a>
                            <a href="setup_wizard.php" class="list-group-item list-group-item-action">
                                <i class="fa fa-magic mr-2"></i> Setup Wizard
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Floating Sync Button -->
    <form method="post" class="sync-button">
        <button type="submit" name="sync_all" class="btn btn-warning btn-lg rounded-circle shadow-lg" 
                style="width: 60px; height: 60px;" title="Sync All Systems">
            <i class="fa fa-sync-alt"></i>
        </button>
    </form>
    
    <script src="js/jquery-1.12.4.js"></script>
    <script src="js/bootstrap.min.js"></script>
    <script>
        // Auto-refresh dashboard every 30 seconds
        setTimeout(function() {
            window.location.reload();
        }, 30000);
        
        // Animate stats on hover
        $(document).ready(function() {
            $('.stat-number').each(function() {
                var $this = $(this);
                var countTo = parseInt($this.text());
                $({ countNum: 0 }).animate({
                    countNum: countTo
                }, {
                    duration: 1000,
                    easing: 'swing',
                    step: function() {
                        $this.text(Math.floor(this.countNum));
                    }
                });
            });
        });
    </script>
</body>
</html>