<?php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (!isset($_SESSION['SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';

// QuickBooks OAuth Configuration
define('QUICKBOOKS_CLIENT_ID', 'your_client_id');
define('QUICKBOOKS_CLIENT_SECRET', 'your_client_secret');
define('QUICKBOOKS_REDIRECT_URI', 'https://yourdomain.com/app/quickbooks_callback.php');
define('QUICKBOOKS_BASE_URL', 'https://sandbox-quickbooks.api.intuit.com');
define('QUICKBOOKS_AUTH_URL', 'https://appcenter.intuit.com/connect/oauth2');
define('QUICKBOOKS_TOKEN_URL', 'https://oauth.platform.intuit.com/oauth2/v1/tokens/bearer');

// Ensure tables exist
ensureQuickBooksTables();

function ensureQuickBooksTables() {
    global $con;
    
    $tables = [
        "CREATE TABLE IF NOT EXISTS quickbooks_tokens (
            id INT AUTO_INCREMENT PRIMARY KEY,
            access_token TEXT NOT NULL,
            refresh_token TEXT NOT NULL,
            realm_id VARCHAR(100) NOT NULL,
            company_id VARCHAR(100),
            expires_at DATETIME NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )",
        
        "CREATE TABLE IF NOT EXISTS integration_sync_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            integration_type ENUM('quickbooks', 'guru', 'gusto') NOT NULL,
            action_type VARCHAR(50) NOT NULL,
            entity_type VARCHAR(50),
            entity_id VARCHAR(100),
            status ENUM('success', 'error', 'warning') NOT NULL,
            message TEXT,
            request_data TEXT,
            response_data TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )"
    ];
    
    foreach ($tables as $sql) {
        mysqli_query($con, $sql);
    }
}

// Helper function to log sync activities
function logSyncActivity($type, $action, $status, $message, $entity_type = null, $entity_id = null, $request = null, $response = null) {
    global $con;
    
    $stmt = mysqli_prepare($con, "
        INSERT INTO integration_sync_logs 
        (integration_type, action_type, entity_type, entity_id, status, message, request_data, response_data) 
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $request_json = $request ? json_encode($request) : null;
    $response_json = $response ? json_encode($response) : null;
    
    mysqli_stmt_bind_param($stmt, "ssssssss", $type, $action, $entity_type, $entity_id, $status, $message, $request_json, $response_json);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
}

// Get QuickBooks access token
function getQuickBooksAccessToken() {
    global $con;
    
    $query = "SELECT * FROM quickbooks_tokens ORDER BY id DESC LIMIT 1";
    $result = mysqli_query($con, $query);
    
    if ($result && mysqli_num_rows($result) > 0) {
        $token = mysqli_fetch_assoc($result);
        
        // Check if token is expired
        $expires_time = strtotime($token['expires_at']);
        $current_time = time();
        
        if ($current_time < $expires_time - 300) {
            return $token;
        }
        
        // Token expired, refresh it
        return refreshQuickBooksToken($token['refresh_token'], $token['realm_id']);
    }
    
    return null;
}

// Refresh QuickBooks token
function refreshQuickBooksToken($refresh_token, $realm_id) {
    $ch = curl_init(QUICKBOOKS_TOKEN_URL);
    
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query([
            'grant_type' => 'refresh_token',
            'refresh_token' => $refresh_token
        ]),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/x-www-form-urlencoded',
            'Accept: application/json',
            'Authorization: Basic ' . base64_encode(QUICKBOOKS_CLIENT_ID . ':' . QUICKBOOKS_CLIENT_SECRET)
        ]
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($http_code == 200) {
        $data = json_decode($response, true);
        
        // Save new token
        global $con;
        $expires_at = date('Y-m-d H:i:s', time() + $data['expires_in']);
        
        $stmt = mysqli_prepare($con, "
            INSERT INTO quickbooks_tokens 
            (access_token, refresh_token, realm_id, expires_at) 
            VALUES (?, ?, ?, ?)
        ");
        
        mysqli_stmt_bind_param($stmt, "ssss", 
            $data['access_token'], 
            $data['refresh_token'], 
            $realm_id, 
            $expires_at
        );
        
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        
        logSyncActivity('quickbooks', 'token_refresh', 'success', 'Token refreshed successfully');
        
        return [
            'access_token' => $data['access_token'],
            'refresh_token' => $data['refresh_token'],
            'realm_id' => $realm_id,
            'expires_at' => $expires_at
        ];
    }
    
    logSyncActivity('quickbooks', 'token_refresh', 'error', "Failed to refresh token: HTTP $http_code");
    return null;
}

// QuickBooks API Request
function quickbooks_request($method, $endpoint, $data = null) {
    $token = getQuickBooksAccessToken();
    
    if (!$token) {
        return ['error' => 'No valid token', 'http_code' => 401];
    }
    
    $url = QUICKBOOKS_BASE_URL . '/v3/company/' . $token['realm_id'] . $endpoint;
    
    $ch = curl_init($url);
    $headers = [
        'Authorization: Bearer ' . $token['access_token'],
        'Accept: application/json',
        'Content-Type: application/json'
    ];
    
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CUSTOMREQUEST => $method,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30
    ]);
    
    if ($data) {
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return [
        'http_code' => $http_code,
        'response' => json_decode($response, true),
        'raw_response' => $response
    ];
}

// Sync employee to QuickBooks as vendor
function syncEmployeeToQuickBooks($employee_id) {
    global $con;
    
    // Get employee data
    $query = "SELECT u.*, im.external_id as qb_vendor_id 
              FROM usermst u 
              LEFT JOIN integration_mappings im ON im.local_employee_id = u.id 
                AND im.integration_type = 'quickbooks'
              WHERE u.id = $employee_id";
    
    $result = mysqli_query($con, $query);
    if (!$result || mysqli_num_rows($result) == 0) {
        return ['success' => false, 'error' => 'Employee not found'];
    }
    
    $employee = mysqli_fetch_assoc($result);
    
    // Prepare vendor data for QuickBooks
    $name_parts = explode(' ', $employee['vName'], 2);
    $first_name = $name_parts[0] ?? '';
    $last_name = $name_parts[1] ?? '';
    
    $vendor_data = [
        'DisplayName' => $employee['vName'],
        'GivenName' => $first_name,
        'FamilyName' => $last_name,
        'PrimaryEmailAddr' => [
            'Address' => $employee['vEmail'] ?? ''
        ],
        'PrimaryPhone' => [
            'FreeFormNumber' => $employee['phone'] ?? ''
        ],
        'Vendor1099' => true,
        'BillAddr' => [
            'Line1' => $employee['address'] ?? '',
            'City' => $employee['city'] ?? '',
            'Country' => 'US',
            'CountrySubDivisionCode' => $employee['state'] ?? 'CA',
            'PostalCode' => $employee['zip'] ?? ''
        ]
    ];
    
    $endpoint = $employee['qb_vendor_id'] 
        ? '/vendor?operation=update' 
        : '/vendor';
    
    if ($employee['qb_vendor_id']) {
        $vendor_data['Id'] = $employee['qb_vendor_id'];
        $vendor_data['sparse'] = true;
    }
    
    $response = quickbooks_request('POST', $endpoint, $vendor_data);
    
    if ($response['http_code'] == 200 || $response['http_code'] == 201) {
        $vendor_id = $response['response']['Vendor']['Id'] ?? null;
        
        if ($vendor_id) {
            // Update or create mapping
            if ($employee['qb_vendor_id']) {
                mysqli_query($con, "
                    UPDATE integration_mappings 
                    SET external_id = '$vendor_id',
                        last_sync_at = NOW()
                    WHERE local_employee_id = $employee_id 
                    AND integration_type = 'quickbooks'
                ");
            } else {
                mysqli_query($con, "
                    INSERT INTO integration_mappings 
                    (integration_type, local_employee_id, external_id, external_name, sync_direction)
                    VALUES ('quickbooks', $employee_id, '$vendor_id', '{$employee['vName']}', 'both')
                ");
            }
            
            // Update usermst with vendor ID
            mysqli_query($con, "
                UPDATE usermst 
                SET quickbooks_vendor_id = '$vendor_id'
                WHERE id = $employee_id
            ");
            
            logSyncActivity('quickbooks', 'vendor_sync', 'success', 
                "Employee {$employee['vName']} synced as vendor", 
                'employee', $employee_id, $vendor_data, $response['response']);
            
            return ['success' => true, 'vendor_id' => $vendor_id];
        }
    }
    
    logSyncActivity('quickbooks', 'vendor_sync', 'error', 
        "Failed to sync employee {$employee['vName']}", 
        'employee', $employee_id, $vendor_data, $response);
    
    return ['success' => false, 'error' => 'API Error', 'response' => $response];
}

// Create bill in QuickBooks for payroll
function createQuickBooksBill($payroll_data, $employee_data) {
    global $con;
    
    // Get or sync vendor
    $vendor_id = $employee_data['quickbooks_vendor_id'];
    if (!$vendor_id) {
        $sync_result = syncEmployeeToQuickBooks($employee_data['id']);
        if (!$sync_result['success']) {
            return $sync_result;
        }
        $vendor_id = $sync_result['vendor_id'];
    }
    
    // Create bill with line items
    $line_items = [];
    
    // Add regular hours line
    if ($payroll_data['regular_pay'] > 0) {
        $line_items[] = [
            'DetailType' => 'AccountBasedExpenseLineDetail',
            'Amount' => $payroll_data['regular_pay'],
            'Description' => "Regular hours: {$payroll_data['hours']} hours @ \${$payroll_data['hourly_rate']}/hr",
            'AccountBasedExpenseLineDetail' => [
                'AccountRef' => [
                    'value' => '67' // Payroll Expenses account ID
                ]
            ]
        ];
    }
    
    // Add bonus line
    if ($payroll_data['solo_bonus_pay'] > 0) {
        $line_items[] = [
            'DetailType' => 'AccountBasedExpenseLineDetail',
            'Amount' => $payroll_data['solo_bonus_pay'],
            'Description' => "Solo bonus: {$payroll_data['solo_hours']} hours",
            'AccountBasedExpenseLineDetail' => [
                'AccountRef' => [
                    'value' => '67' // Payroll Expenses account ID
                ]
            ]
        ];
    }
    
    // Add custom fields lines
    if (!empty($payroll_data['custom_fields'])) {
        foreach ($payroll_data['custom_fields'] as $field) {
            $line_items[] = [
                'DetailType' => 'AccountBasedExpenseLineDetail',
                'Amount' => $field['calculated_pay'],
                'Description' => "{$field['field_name']}: {$field['field_value']} {$field['unit_label']}",
                'AccountBasedExpenseLineDetail' => [
                    'AccountRef' => [
                        'value' => '67' // Payroll Expenses account ID
                    ]
                ]
            ];
        }
    }
    
    // Create bill
    $bill_data = [
        'VendorRef' => [
            'value' => $vendor_id
        ],
        'TxnDate' => $payroll_data['pay_date'],
        'DueDate' => $payroll_data['pay_date'],
        'Line' => $line_items,
        'PrivateNote' => "Payroll for week {$payroll_data['week_start']}",
        'DocNumber' => "PAY-{$payroll_data['week_start']}-{$employee_data['id']}"
    ];
    
    $response = quickbooks_request('POST', '/purchase', $bill_data);
    
    if ($response['http_code'] == 200 || $response['http_code'] == 201) {
        $bill_id = $response['response']['Purchase']['Id'] ?? null;
        
        logSyncActivity('quickbooks', 'bill_create', 'success', 
            "Bill created for {$employee_data['vName']}", 
            'payroll', $payroll_data['id'], $bill_data, $response['response']);
        
        return ['success' => true, 'bill_id' => $bill_id];
    }
    
    logSyncActivity('quickbooks', 'bill_create', 'error', 
        "Failed to create bill for {$employee_data['vName']}", 
        'payroll', $payroll_data['id'], $bill_data, $response);
    
    return ['success' => false, 'error' => 'API Error', 'response' => $response];
}

// Sync payroll to QuickBooks
function syncPayrollToQuickBooks($payroll_uuid) {
    global $con;
    
    // Get payroll data with reports
    $query = "
        SELECT p.*, wr.*, u.*
        FROM payrolls p
        JOIN weekly_reports wr ON wr.gusto_payroll_uuid = p.payroll_uuid
        JOIN usermst u ON u.id = wr.employee_id
        WHERE p.payroll_uuid = '" . mysqli_real_escape_string($con, $payroll_uuid) . "'
        AND wr.gusto_submitted = 1
        AND wr.quickbooks_synced = 0
    ";
    
    $result = mysqli_query($con, $query);
    $sync_results = [];
    
    while ($report = mysqli_fetch_assoc($result)) {
        // Get custom fields for this report
        $cf_query = mysqli_query($con, "
            SELECT cpf.field_name, cpf.unit_label, wrf.field_value, wrf.calculated_pay
            FROM weekly_report_custom_fields wrf
            JOIN custom_payroll_fields cpf ON cpf.id = wrf.custom_field_id
            WHERE wrf.weekly_report_id = {$report['id']}
        ");
        
        $custom_fields = [];
        while ($cf = mysqli_fetch_assoc($cf_query)) {
            $custom_fields[] = $cf;
        }
        
        // Prepare payroll data
        $payroll_data = [
            'id' => $report['id'],
            'week_start' => $report['week_start'],
            'pay_date' => $report['pay_date'],
            'hours' => $report['hours_total'],
            'hourly_rate' => $report['hourly_rate'],
            'regular_pay' => $report['regular_pay'],
            'solo_hours' => $report['solo_hours'],
            'solo_bonus_pay' => $report['solo_bonus_pay'],
            'custom_fields' => $custom_fields
        ];
        
        // Create bill in QuickBooks
        $result = createQuickBooksBill($payroll_data, $report);
        
        if ($result['success']) {
            // Mark as synced
            mysqli_query($con, "
                UPDATE weekly_reports 
                SET quickbooks_synced = 1, 
                    quickbooks_ref = '{$result['bill_id']}',
                    updated_at = NOW()
                WHERE id = {$report['id']}
            ");
            
            $sync_results[] = [
                'employee' => $report['vName'],
                'status' => 'success',
                'bill_id' => $result['bill_id']
            ];
        } else {
            $sync_results[] = [
                'employee' => $report['vName'],
                'status' => 'error',
                'error' => $result['error']
            ];
        }
    }
    
    return $sync_results;
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['connect_quickbooks'])) {
        // Start OAuth flow
        $params = [
            'client_id' => QUICKBOOKS_CLIENT_ID,
            'response_type' => 'code',
            'scope' => 'com.intuit.quickbooks.accounting openid profile email phone address',
            'redirect_uri' => QUICKBOOKS_REDIRECT_URI,
            'state' => bin2hex(random_bytes(16))
        ];
        
        $_SESSION['qb_state'] = $params['state'];
        $auth_url = QUICKBOOKS_AUTH_URL . '?' . http_build_query($params);
        
        header('Location: ' . $auth_url);
        exit;
    }
    
    if (isset($_POST['sync_payroll'])) {
        $payroll_uuid = $_POST['payroll_uuid'];
        $results = syncPayrollToQuickBooks($payroll_uuid);
        
        $_SESSION['sync_results'] = $results;
        header('Location: quickbooks_integration.php');
        exit;
    }
    
    if (isset($_POST['sync_all_vendors'])) {
        // Sync all employees as vendors
        $employees = mysqli_query($con, "
            SELECT id, vName FROM usermst 
            WHERE is_active = 1 
            AND gusto_employee_uuid IS NOT NULL
        ");
        
        $results = [];
        while ($emp = mysqli_fetch_assoc($employees)) {
            $result = syncEmployeeToQuickBooks($emp['id']);
            $results[] = [
                'employee' => $emp['vName'],
                'success' => $result['success'],
                'vendor_id' => $result['vendor_id'] ?? null,
                'error' => $result['error'] ?? null
            ];
        }
        
        $_SESSION['vendor_sync_results'] = $results;
        header('Location: quickbooks_integration.php');
        exit;
    }
}

// Get sync status
$token_status = getQuickBooksAccessToken();
$connected = $token_status ? true : false;

$payrolls = mysqli_query($con, "
    SELECT p.*, 
           COUNT(DISTINCT wr.id) as report_count,
           0 as synced_count
    FROM payrolls p
    LEFT JOIN weekly_reports wr ON wr.gusto_payroll_uuid = p.payroll_uuid
    GROUP BY p.id
    ORDER BY p.created_at DESC
    LIMIT 10
");


// Get sync logs
$sync_logs = mysqli_query($con, "
    SELECT * FROM integration_sync_logs 
    WHERE integration_type = 'quickbooks'
    ORDER BY created_at DESC
    LIMIT 20
");
?>

<!DOCTYPE html>
<html>
<head>
    <title>QuickBooks Integration</title>
    <link href="css/bootstrap.min.css" rel="stylesheet">
    <link href="css/font-awesome.css" rel="stylesheet">
    <style>
        .status-connected { color: #28a745; }
        .status-disconnected { color: #dc3545; }
        .sync-card { border-left: 4px solid #28a745; }
        .log-table { font-size: 0.85rem; }
        .log-success { background-color: #d4edda !important; }
        .log-error { background-color: #f8d7da !important; }
        .log-warning { background-color: #fff3cd !important; }
    </style>
</head>
<body>
    <div class="container mt-4">
        <h3>QuickBooks Integration</h3>
        
        <!-- Connection Status -->
        <div class="card mb-4">
            <div class="card-header <?= $connected ? 'bg-success' : 'bg-danger' ?> text-white">
                <h5 class="mb-0">
                    <i class="fa fa-<?= $connected ? 'check-circle' : 'times-circle' ?>"></i>
                    Connection Status: <?= $connected ? 'Connected' : 'Disconnected' ?>
                </h5>
            </div>
            <div class="card-body">
                <?php if($connected): ?>
                    <div class="alert alert-success">
                        <h5><i class="fa fa-check"></i> Connected to QuickBooks</h5>
                        <p class="mb-0">
                            Token expires: <?= date('Y-m-d H:i:s', strtotime($token_status['expires_at'])) ?>
                        </p>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>Sync Vendors</h6>
                                    <p>Sync employees as QuickBooks vendors</p>
                                    <form method="post">
                                        <button name="sync_all_vendors" class="btn btn-primary">
                                            <i class="fa fa-sync"></i> Sync All Employees
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>Test Connection</h6>
                                    <p>Test QuickBooks API connection</p>
                                    <a href="test_quickbooks.php" class="btn btn-info">
                                        <i class="fa fa-plug"></i> Test Connection
                                    </a>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card">
                                <div class="card-body text-center">
                                    <h6>View Mappings</h6>
                                    <p>View employee-vendor mappings</p>
                                    <a href="quickbooks_mappings.php" class="btn btn-secondary">
                                        <i class="fa fa-list"></i> View Mappings
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <h5><i class="fa fa-exclamation-triangle"></i> Not Connected</h5>
                        <p class="mb-3">Connect to QuickBooks to sync payroll data</p>
                        <form method="post">
                            <button name="connect_quickbooks" class="btn btn-success btn-lg">
                                <i class="fa fa-link"></i> Connect to QuickBooks
                            </button>
                        </form>
                    </div>
                    
                    <div class="alert alert-info">
                        <h6><i class="fa fa-info-circle"></i> Setup Instructions:</h6>
                        <ol class="mb-0">
                            <li>Create a QuickBooks Developer Account</li>
                            <li>Create a new app in QuickBooks Developer Portal</li>
                            <li>Set redirect URI to: <?= QUICKBOOKS_REDIRECT_URI ?></li>
                            <li>Update client ID and secret in configuration</li>
                        </ol>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Sync Results -->
        <?php if(isset($_SESSION['sync_results'])): ?>
            <div class="card mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0">Sync Results</h5>
                </div>
                <div class="card-body">
                    <?php foreach($_SESSION['sync_results'] as $result): ?>
                        <div class="alert alert-<?= $result['status'] == 'success' ? 'success' : 'danger' ?>">
                            <strong><?= $result['employee'] ?>:</strong>
                            <?= $result['status'] == 'success' 
                                ? 'Synced successfully (Bill ID: ' . $result['bill_id'] . ')' 
                                : 'Error: ' . ($result['error'] ?? 'Unknown error') ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php unset($_SESSION['sync_results']); ?>
        <?php endif; ?>
        
        <?php if(isset($_SESSION['vendor_sync_results'])): ?>
            <div class="card mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0">Vendor Sync Results</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Employee</th>
                                    <th>Status</th>
                                    <th>Vendor ID</th>
                                    <th>Error</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach($_SESSION['vendor_sync_results'] as $result): ?>
                                    <tr class="<?= $result['success'] ? 'table-success' : 'table-danger' ?>">
                                        <td><?= $result['employee'] ?></td>
                                        <td>
                                            <?= $result['success'] 
                                                ? '<span class="badge badge-success">Success</span>' 
                                                : '<span class="badge badge-danger">Failed</span>' ?>
                                        </td>
                                        <td><?= $result['vendor_id'] ?? '-' ?></td>
                                        <td><?= $result['error'] ?? '-' ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <?php unset($_SESSION['vendor_sync_results']); ?>
        <?php endif; ?>
        
        <!-- Payroll Sync Section -->
        <?php if($connected): ?>
            <div class="card mb-4 sync-card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="fa fa-exchange"></i> Sync Payroll to QuickBooks</h5>
                </div>
                <div class="card-body">
                    <form method="post" class="row">
                        <div class="col-md-6">
                            <label class="form-label">Select Payroll to Sync</label>
                            <select name="payroll_uuid" class="form-control" required>
                                <option value="">Select a payroll...</option>
                                <?php while($payroll = mysqli_fetch_assoc($payrolls)): ?>
                                    <option value="<?= $payroll['payroll_uuid'] ?>">
                                        Week: <?= $payroll['week_start'] ?> 
                                        | Reports: <?= $payroll['report_count'] ?>
                                        | Synced: <?= $payroll['synced_count'] ?>/<?= $payroll['report_count'] ?>
                                        | Status: <?= $payroll['status'] ?>
                                    </option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        <div class="col-md-3 d-flex align-items-end">
                            <button name="sync_payroll" class="btn btn-primary">
                                <i class="fa fa-sync"></i> Sync to QuickBooks
                            </button>
                        </div>
                    </form>
                    <small class="text-muted mt-2">
                        This will create bills in QuickBooks for each employee's payroll. 
                        Employees will be synced as vendors if not already.
                    </small>
                </div>
            </div>
        <?php endif; ?>
        
        <!-- Sync Logs -->
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="mb-0"><i class="fa fa-history"></i> Recent Sync Logs</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm log-table">
                        <thead>
                            <tr>
                                <th>Time</th>
                                <th>Action</th>
                                <th>Entity</th>
                                <th>Status</th>
                                <th>Message</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while($log = mysqli_fetch_assoc($sync_logs)): ?>
                                <tr class="log-<?= $log['status'] ?>">
                                    <td><?= date('Y-m-d H:i', strtotime($log['created_at'])) ?></td>
                                    <td><?= $log['action_type'] ?></td>
                                    <td><?= $log['entity_type'] ?> <?= $log['entity_id'] ?></td>
                                    <td>
                                        <span class="badge badge-<?= $log['status'] ?>">
                                            <?= $log['status'] ?>
                                        </span>
                                    </td>
                                    <td><?= substr($log['message'], 0, 100) ?></td>
                                </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <script src="js/jquery-1.12.4.js"></script>
    <script src="js/bootstrap.min.js"></script>
</body>
</html>