<?php
session_start();
if (!isset($_SESSION['GUSTO_SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';

$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
unset($_SESSION['success'], $_SESSION['error']);

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_field'])) {
        $field_name = mysqli_real_escape_string($con, $_POST['field_name']);
        $field_slug = strtolower(preg_replace('/[^a-zA-Z0-9_]/', '_', $field_name));
        $field_type = $_POST['field_type'];
        $unit_label = mysqli_real_escape_string($con, $_POST['unit_label'] ?? '');
        $default_value = floatval($_POST['default_value'] ?? 0);
        $is_required = isset($_POST['is_required']) ? 1 : 0;
        $calculation_type = $_POST['calculation_type'] ?? 'fixed_amount';
        $rate_field = $_POST['rate_field'] ?? '';
        
        $stmt = mysqli_prepare($con, "
            INSERT INTO custom_payroll_fields 
            (field_name, field_slug, field_type, unit_label, default_value, is_required, calculation_type, rate_field, display_order) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, 
                (SELECT COALESCE(MAX(display_order), 0) + 1 FROM custom_payroll_fields c)
            )
        ");
        mysqli_stmt_bind_param($stmt, "ssssdiss", $field_name, $field_slug, $field_type, $unit_label, $default_value, $is_required, $calculation_type, $rate_field);
        
        if (mysqli_stmt_execute($stmt)) {
            $_SESSION['success'] = "Custom field added successfully!";
        } else {
            $_SESSION['error'] = "Failed to add field: " . mysqli_error($con);
        }
        mysqli_stmt_close($stmt);
    }
    
    if (isset($_POST['update_field'])) {
        $field_id = intval($_POST['field_id']);
        $field_name = mysqli_real_escape_string($con, $_POST['field_name']);
        $field_type = $_POST['field_type'];
        $unit_label = mysqli_real_escape_string($con, $_POST['unit_label'] ?? '');
        $default_value = floatval($_POST['default_value'] ?? 0);
        $is_required = isset($_POST['is_required']) ? 1 : 0;
        $is_active = isset($_POST['is_active']) ? 1 : 0;
        $display_order = intval($_POST['display_order'] ?? 0);
        $calculation_type = $_POST['calculation_type'] ?? 'fixed_amount';
        $rate_field = $_POST['rate_field'] ?? '';
        
        $stmt = mysqli_prepare($con, "
            UPDATE custom_payroll_fields 
            SET field_name = ?, field_type = ?, unit_label = ?, default_value = ?, 
                is_required = ?, is_active = ?, display_order = ?, calculation_type = ?, rate_field = ?
            WHERE id = ?
        ");
        mysqli_stmt_bind_param($stmt, "sssdiiissi", $field_name, $field_type, $unit_label, 
            $default_value, $is_required, $is_active, $display_order, $calculation_type, $rate_field, $field_id);
        
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
        $_SESSION['success'] = "Field updated successfully!";
    }
    
    if (isset($_POST['delete_field'])) {
        $field_id = intval($_POST['field_id']);
        
        // Check if field has data
        $check = mysqli_query($con, "SELECT COUNT(*) as count FROM weekly_report_custom_fields WHERE custom_field_id = $field_id");
        $data = mysqli_fetch_assoc($check);
        
        if ($data['count'] == 0) {
            mysqli_query($con, "DELETE FROM custom_payroll_fields WHERE id = $field_id");
            $_SESSION['success'] = "Field deleted successfully!";
        } else {
            $_SESSION['error'] = "Cannot delete field with existing data. Deactivate instead.";
        }
    }
    
    header('Location: custom_payroll_fields.php');
    exit;
}

// Fetch all custom fields
$fields = mysqli_query($con, "
    SELECT * FROM custom_payroll_fields 
    ORDER BY display_order, field_name
");
?>

<?php include('header.php'); ?>

<h3>Custom Payroll Fields</h3>

<?php if($success_message): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <?= $success_message ?>
        <button type="button" class="close" data-dismiss="alert">&times;</button>
    </div>
<?php endif; ?>

<?php if($error_message): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <?= $error_message ?>
        <button type="button" class="close" data-dismiss="alert">&times;</button>
    </div>
<?php endif; ?>

<!-- Add New Field Form -->
<div class="card field-card mb-4">
    <div class="card-header bg-primary text-white">
        <h5 class="mb-0"><i class="fas fa-plus-circle"></i> Add New Payroll Field</h5>
    </div>
    <div class="card-body">
        <form method="post" class="row g-3">
            <div class="col-md-3">
                <label class="form-label">Field Name *</label>
                <input type="text" name="field_name" class="form-control" required>
                <small class="text-muted">e.g., "Bonus Hours", "Commission"</small>
            </div>
            <div class="col-md-2">
                <label class="form-label">Field Type *</label>
                <select name="field_type" class="form-control" required>
                    <option value="hours">Hours</option>
                    <option value="dollars">Dollars</option>
                    <option value="miles">Miles</option>
                    <option value="units">Units</option>
                    <option value="percentage">Percentage</option>
                    <option value="number">Number</option>
                    <option value="text">Text</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Calculation Type *</label>
                <select name="calculation_type" class="form-control" required>
                    <option value="fixed_amount">Fixed Amount</option>
                    <option value="multiply_rate">Multiply by Rate</option>
                    <option value="percentage_of_gross">Percentage of Gross</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Rate Field</label>
                <select name="rate_field" class="form-control">
                    <option value="">-- Select Rate --</option>
                    <option value="hourly_rate">Hourly Rate</option>
                    <option value="solo_rate">Solo Rate</option>
                    <option value="miles_rate">Miles Rate</option>
                    <option value="miscellaneous_rate">Miscellaneous Rate</option>
                </select>
                <small class="text-muted">For "Multiply by Rate" type</small>
            </div>
            <div class="col-md-2">
                <label class="form-label">Unit Label</label>
                <input type="text" name="unit_label" class="form-control" 
                       placeholder="e.g., hours, miles, $">
            </div>
            <div class="col-md-1 d-flex align-items-end">
                <div class="form-check">
                    <input type="checkbox" name="is_required" class="form-check-input">
                    <label class="form-check-label">Required</label>
                </div>
            </div>
            <div class="col-md-12 mt-3">
                <div class="row">
                    <div class="col-md-2">
                        <label class="form-label">Default Value</label>
                        <input type="number" step="0.01" name="default_value" 
                               class="form-control" value="0">
                    </div>
                    <div class="col-md-10 d-flex align-items-end">
                        <button name="add_field" class="btn btn-success">
                            <i class="fas fa-plus"></i> Add Field
                        </button>
                    </div>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Fields List -->
<div class="card">
    <div class="card-header bg-light d-flex justify-content-between align-items-center">
        <h5 class="mb-0"><i class="fas fa-list"></i> Existing Custom Fields</h5>
        <small>Drag to reorder</small>
    </div>
    <div class="card-body">
        <?php if (mysqli_num_rows($fields) > 0): ?>
            <ul class="sortable-list" id="sortableFields">
                <?php while($field = mysqli_fetch_assoc($fields)): 
                    $calc_type_labels = [
                        'fixed_amount' => 'Fixed Amount',
                        'multiply_rate' => 'Multiply by Rate',
                        'percentage_of_gross' => 'Percentage of Gross'
                    ];
                ?>
                    <li class="sortable-item mb-3 <?= $field['is_active'] ? 'active-field' : 'inactive-field' ?>" 
                        data-id="<?= $field['id'] ?>">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="flex-grow-1">
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <i class="fas fa-grip-vertical text-muted handle" style="cursor: move;"></i>
                                    </div>
                                    <div>
                                        <strong><?= htmlspecialchars($field['field_name']) ?></strong>
                                        <span class="badge badge-info field-type-badge ml-2">
                                            <?= $field['field_type'] ?>
                                        </span>
                                        <?php if($field['unit_label']): ?>
                                            <span class="unit-label ml-2">(<?= $field['unit_label'] ?>)</span>
                                        <?php endif; ?>
                                        <span class="badge badge-secondary ml-2">
                                            <?= $calc_type_labels[$field['calculation_type']] ?? $field['calculation_type'] ?>
                                        </span>
                                        <?php if($field['rate_field']): ?>
                                            <span class="badge badge-warning ml-1">
                                                Rate: <?= $field['rate_field'] ?>
                                            </span>
                                        <?php endif; ?>
                                        <?php if($field['is_required']): ?>
                                            <span class="badge badge-danger ml-1">Required</span>
                                        <?php endif; ?>
                                        <?php if(!$field['is_active']): ?>
                                            <span class="badge badge-secondary ml-1">Inactive</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php if($field['default_value'] != 0): ?>
                                    <div class="mt-1">
                                        <small class="text-muted">
                                            Default: <?= $field['default_value'] ?>
                                            <?php if($field['unit_label']): ?>
                                                <?= $field['unit_label'] ?>
                                            <?php endif; ?>
                                        </small>
                                    </div>
                                <?php endif; ?>
                            </div>
                            <div class="flex-shrink-0">
                                <button class="btn btn-sm btn-info" data-toggle="modal" 
                                        data-target="#editFieldModal<?= $field['id'] ?>">
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                <?php if($field['is_active']): ?>
                                    <form method="post" class="d-inline">
                                        <input type="hidden" name="field_id" value="<?= $field['id'] ?>">
                                        <button name="delete_field" class="btn btn-sm btn-danger"
                                                onclick="return confirm('Delete this field? This cannot be undone.')">
                                            <i class="fas fa-trash"></i> Delete
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <!-- Edit Modal -->
                        <div class="modal fade" id="editFieldModal<?= $field['id'] ?>" tabindex="-1" role="dialog">
                            <div class="modal-dialog modal-lg" role="document">
                                <div class="modal-content">
                                    <div class="modal-header bg-primary text-white">
                                        <h5 class="modal-title">Edit Custom Field</h5>
                                        <button type="button" class="close text-white" data-dismiss="modal">
                                            &times;
                                        </button>
                                    </div>
                                    <form method="post">
                                        <div class="modal-body">
                                            <input type="hidden" name="field_id" value="<?= $field['id'] ?>">
                                            
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Field Name *</label>
                                                        <input type="text" name="field_name" 
                                                               value="<?= htmlspecialchars($field['field_name']) ?>" 
                                                               class="form-control" required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Field Type *</label>
                                                        <select name="field_type" class="form-control" required>
                                                            <?php 
                                                            $types = ['hours', 'dollars', 'miles', 'units', 'percentage', 'number', 'text'];
                                                            foreach($types as $type): ?>
                                                                <option value="<?= $type ?>" 
                                                                    <?= $field['field_type'] == $type ? 'selected' : '' ?>>
                                                                    <?= ucfirst($type) ?>
                                                                </option>
                                                            <?php endforeach; ?>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Calculation Type *</label>
                                                        <select name="calculation_type" class="form-control" required>
                                                            <option value="fixed_amount" <?= $field['calculation_type'] == 'fixed_amount' ? 'selected' : '' ?>>Fixed Amount</option>
                                                            <option value="multiply_rate" <?= $field['calculation_type'] == 'multiply_rate' ? 'selected' : '' ?>>Multiply by Rate</option>
                                                            <option value="percentage_of_gross" <?= $field['calculation_type'] == 'percentage_of_gross' ? 'selected' : '' ?>>Percentage of Gross</option>
                                                        </select>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Rate Field</label>
                                                        <select name="rate_field" class="form-control">
                                                            <option value="">-- Select Rate --</option>
                                                            <option value="hourly_rate" <?= $field['rate_field'] == 'hourly_rate' ? 'selected' : '' ?>>Hourly Rate</option>
                                                            <option value="solo_rate" <?= $field['rate_field'] == 'solo_rate' ? 'selected' : '' ?>>Solo Rate</option>
                                                            <option value="miles_rate" <?= $field['rate_field'] == 'miles_rate' ? 'selected' : '' ?>>Miles Rate</option>
                                                            <option value="miscellaneous_rate" <?= $field['rate_field'] == 'miscellaneous_rate' ? 'selected' : '' ?>>Miscellaneous Rate</option>
                                                        </select>
                                                        <small class="text-muted">Only for "Multiply by Rate" calculation type</small>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Unit Label</label>
                                                        <input type="text" name="unit_label" 
                                                               value="<?= htmlspecialchars($field['unit_label']) ?>" 
                                                               class="form-control">
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label class="form-label">Default Value</label>
                                                        <input type="number" step="0.01" name="default_value" 
                                                               value="<?= $field['default_value'] ?>" 
                                                               class="form-control">
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="row">
                                                <div class="col-md-4">
                                                    <div class="form-group">
                                                        <label class="form-label">Display Order</label>
                                                        <input type="number" name="display_order" 
                                                               value="<?= $field['display_order'] ?>" 
                                                               class="form-control">
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-check mt-4">
                                                        <input type="checkbox" name="is_required" 
                                                               class="form-check-input" 
                                                               <?= $field['is_required'] ? 'checked' : '' ?>>
                                                        <label class="form-check-label">Required Field</label>
                                                    </div>
                                                </div>
                                                <div class="col-md-4">
                                                    <div class="form-check mt-4">
                                                        <input type="checkbox" name="is_active" 
                                                               class="form-check-input" 
                                                               <?= $field['is_active'] ? 'checked' : '' ?>>
                                                        <label class="form-check-label">Active</label>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" 
                                                    data-dismiss="modal">Cancel</button>
                                            <button name="update_field" class="btn btn-primary">
                                                <i class="fas fa-save"></i> Save Changes
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    </li>
                <?php endwhile; ?>
            </ul>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="fas fa-list-alt fa-4x text-muted mb-3"></i>
                <h4>No Custom Fields Yet</h4>
                <p class="text-muted">Add your first custom field using the form above.</p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Instructions -->
<div class="card mt-4">
    <div class="card-header bg-light">
        <h5 class="mb-0"><i class="fas fa-info-circle"></i> How Custom Fields Work</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-4">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6><i class="fas fa-calculator text-primary"></i> Fixed Amount</h6>
                        <p class="small mb-0">Field value is used as-is. Example: $50 bonus = $50</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6><i class="fas fa-times-circle text-success"></i> Multiply by Rate</h6>
                        <p class="small mb-0">Value × Rate. Example: 10 hours × $20/hour = $200</p>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card bg-light">
                    <div class="card-body">
                        <h6><i class="fas fa-percentage text-warning"></i> Percentage of Gross</h6>
                        <p class="small mb-0">Percentage of total gross pay. Example: 5% of $1000 = $50</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include('footer.php'); ?>

<style>
.field-card { margin-bottom: 15px; }
.field-type-badge { font-size: 0.7rem; }
.active-field { 
    border-left: 4px solid #28a745;
    background: #f8fff9;
    padding: 15px;
    border-radius: 5px;
}
.inactive-field { 
    border-left: 4px solid #dc3545;
    background: #fff8f8;
    padding: 15px;
    border-radius: 5px;
    opacity: 0.7;
}
.unit-label { color: #6c757d; font-size: 0.9rem; }
.sortable-list { list-style: none; padding: 0; }
.sortable-item { 
    cursor: move;
    transition: all 0.3s;
}
.sortable-item:hover {
    transform: translateX(5px);
}
.handle:hover {
    color: #007bff !important;
}
</style>

<script src="https://code.jquery.com/jquery-3.5.1.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@4.5.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://code.jquery.com/ui/1.12.1/jquery-ui.min.js"></script>
<script>
$(function() {
    // Make fields sortable
    $("#sortableFields").sortable({
        handle: ".handle",
        update: function(event, ui) {
            var order = [];
            $("#sortableFields .sortable-item").each(function(index) {
                order.push({
                    id: $(this).data('id'),
                    position: index + 1
                });
            });
            
            // Save order to database
            $.post('update_field_order.php', { order: order }, function(response) {
                console.log('Field order updated');
                // Show success message
                showToast('Field order saved!');
            });
        }
    });
    
    // Show/hide rate field based on calculation type
    $('select[name="calculation_type"]').change(function() {
        var calculationType = $(this).val();
        var rateField = $(this).closest('form').find('select[name="rate_field"]');
        
        if (calculationType === 'multiply_rate') {
            rateField.closest('.form-group').show();
            rateField.prop('required', true);
        } else {
            rateField.closest('.form-group').hide();
            rateField.prop('required', false);
        }
    });
    
    // Initialize on page load
    $('select[name="calculation_type"]').trigger('change');
    
    // Toast notification function
    function showToast(message) {
        var toast = $('<div class="toast" style="position: fixed; bottom: 20px; right: 20px; z-index: 9999;">' +
            '<div class="alert alert-success alert-dismissible">' +
            '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
            message +
            '</div></div>');
        $('body').append(toast);
        setTimeout(function() {
            toast.fadeOut();
        }, 3000);
    }
});
</script>