<?php
session_start();
if (!isset($_SESSION['GUSTO_SESS_MEMBER_ID'])) { 
    header('Location: index.php'); 
    exit; 
}
require_once 'db.php';
require_once 'gusto_api.php';
$config = require __DIR__ . '/gusto_config.php';

// Debug: Check database connection
if (!$con) {
    die("Database connection failed: " . mysqli_connect_error());
}

// Store flash messages before any output
$success_message = $_SESSION['success'] ?? '';
$error_message = $_SESSION['error'] ?? '';
$warning_message = $_SESSION['warning'] ?? '';

// Clear flash messages after storing
unset($_SESSION['success'], $_SESSION['error'], $_SESSION['warning']);

// Function to ensure database schema
function ensureDatabaseSchema($con) {
    // Check and add usermst columns
    $usermst_columns = [
        ['table' => 'usermst', 'name' => 'miscellaneous_rate', 'type' => 'DECIMAL(10,2) DEFAULT 1.00']
    ];
    
    // Check and add weekly_reports columns  
    $weekly_reports_columns = [
        ['table' => 'weekly_reports', 'name' => 'miscellaneous', 'type' => 'DECIMAL(10,2) DEFAULT 0.00', 'after' => 'miles'],
        ['table' => 'weekly_reports', 'name' => 'miscellaneous_pay', 'type' => 'DECIMAL(10,2) DEFAULT 0.00', 'after' => 'mileage_pay']
    ];
    
    $all_columns = array_merge($usermst_columns, $weekly_reports_columns);
    
    foreach ($all_columns as $column) {
        $check_query = "SELECT COUNT(*) as count FROM information_schema.COLUMNS 
                       WHERE TABLE_SCHEMA = DATABASE() 
                       AND TABLE_NAME = '{$column['table']}' 
                       AND COLUMN_NAME = '{$column['name']}'";
        
        $result = mysqli_query($con, $check_query);
        if ($result) {
            $exists = mysqli_fetch_assoc($result)['count'];
            
            if (!$exists) {
                $alter_query = "ALTER TABLE {$column['table']} 
                               ADD COLUMN {$column['name']} {$column['type']}";
                
                // Add AFTER clause if specified
                if (isset($column['after'])) {
                    $alter_query .= " AFTER {$column['after']}";
                }
                
                if (!mysqli_query($con, $alter_query)) {
                    error_log("Failed to add column {$column['name']} to {$column['table']}: " . mysqli_error($con));
                }
            }
        }
    }
}

// Call the function to ensure schema
ensureDatabaseSchema($con);

// Set default values if not set
$update_queries = [
    "UPDATE usermst SET is_active = 1 WHERE is_active IS NULL",
    "UPDATE usermst SET dAmount = 16.00 WHERE dAmount IS NULL OR dAmount = 0.00",
    "UPDATE usermst SET solo_rate = 3.00 WHERE solo_rate IS NULL OR solo_rate = 0.00",
    "UPDATE usermst SET miles_rate = 0.25 WHERE miles_rate IS NULL OR miles_rate = 0.00",
    "UPDATE usermst SET miscellaneous_rate = 1.00 WHERE miscellaneous_rate IS NULL OR miscellaneous_rate = 0.00"
];

foreach ($update_queries as $query) {
    $result = mysqli_query($con, $query);
    if (!$result) {
        error_log("UPDATE failed for query: $query - Error: " . mysqli_error($con));
    }
}

// Pagination setup
$records_per_page = 15;
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max(1, $page);
$offset = ($page - 1) * $records_per_page;

// Search and filter setup
$search = isset($_GET['search']) ? mysqli_real_escape_string($con, $_GET['search']) : '';
$status_filter = isset($_GET['status']) ? $_GET['status'] : 'active';

// Build WHERE clause
$where_conditions = [];
if (!empty($search)) {
    $where_conditions[] = "(u.vName LIKE '%$search%' OR u.vEmail LIKE '%$search%')";
}
if ($status_filter === 'active') {
    $where_conditions[] = "u.is_active = 1";
} elseif ($status_filter === 'inactive') {
    $where_conditions[] = "u.is_active = 0";
}
$where_clause = $where_conditions ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get total records for pagination
$count_query = "SELECT COUNT(*) as total FROM usermst u $where_clause";
$count_result = mysqli_query($con, $count_query);
if (!$count_result) {
    die("Count query failed: " . mysqli_error($con));
}
$total_records = mysqli_fetch_assoc($count_result)['total'];
$total_pages = ceil($total_records / $records_per_page);

// Handle manual mapping
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_map'])) {
    $emp_id = intval($_POST['emp_id']);
    $gusto_uuid = trim($_POST['gusto_uuid']);
    $stmt = mysqli_prepare($con, "UPDATE usermst SET gusto_employee_uuid = ? WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "si", $gusto_uuid, $emp_id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    $_SESSION['success'] = "Mapping saved successfully.";
    header('Location: gusto_employees.php' . ($page > 1 ? "?page=$page" : "")); 
    exit();
}

// Handle rate updates INCLUDING miscellaneous
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_rates'])) {
    $emp_id = intval($_POST['emp_id']);
    $hourly_rate = floatval($_POST['hourly_rate']);
    $solo_rate = floatval($_POST['solo_rate']);
    $custom_miles_rate = !empty($_POST['custom_miles_rate']) ? floatval($_POST['custom_miles_rate']) : null;
    $miscellaneous_rate = floatval($_POST['miscellaneous_rate'] ?? 1.00);
    
    // Use miles_rate column instead of custom_miles_rate for consistency
    $update_query = "UPDATE usermst SET dAmount = ?, solo_rate = ?, miles_rate = ?, miscellaneous_rate = ? WHERE id = ?";
    $stmt = mysqli_prepare($con, $update_query);
    mysqli_stmt_bind_param($stmt, "ddddi", $hourly_rate, $solo_rate, $custom_miles_rate, $miscellaneous_rate, $emp_id);
    $result = mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    
    if ($result) {
        $_SESSION['success'] = "Rates updated successfully for employee ID: $emp_id";
    } else {
        $_SESSION['error'] = "Failed to update rates. Error: " . mysqli_error($con);
    }
    
    header('Location: gusto_employees.php' . ($page > 1 ? "?page=$page" : "")); 
    exit();
}

// Create new Gusto employee
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_gusto'])) {
    $emp_id = intval($_POST['emp_id_create']);
    $res = mysqli_query($con, "SELECT * FROM usermst WHERE id = {$emp_id} LIMIT 1");
    
    if ($res && mysqli_num_rows($res) > 0) {
        $r = mysqli_fetch_assoc($res);
        
        $name_parts = explode(' ', $r['vName'] ?? 'No Name');
        $first_name = $name_parts[0] ?? 'NoName';
        $last_name = count($name_parts) > 1 ? implode(' ', array_slice($name_parts, 1)) : 'Employee';
        
        $payload = [
            'first_name' => $first_name,
            'last_name'  => $last_name,
            'email'      => $r['vEmail'] ?? null,
            'date_of_hire' => date('Y-m-d')
        ];
        
        $resp = gusto_request('POST', "companies/{$config->COMPANY_UUID}/employees", $payload);
        if ($resp['http_code'] >= 200 && $resp['http_code'] < 300 && !empty($resp['json']['uuid'])) {
            $gusto_uuid = $resp['json']['uuid'];
            $stmt = mysqli_prepare($con, "UPDATE usermst SET gusto_employee_uuid = ? WHERE id = ?");
            mysqli_stmt_bind_param($stmt, "si", $gusto_uuid, $emp_id);
            mysqli_stmt_execute($stmt);
            mysqli_stmt_close($stmt);
            $_SESSION['success'] = "Employee created in Gusto successfully! UUID: " . substr($gusto_uuid, 0, 8) . "...";
        } else {
            $error_msg = "Create failed: ";
            if (isset($resp['json']['errors'])) {
                foreach ($resp['json']['errors'] as $error) {
                    $error_msg .= $error['message'] . " ";
                }
            } else {
                $error_msg .= "HTTP Code: " . $resp['http_code'];
            }
            $_SESSION['error'] = $error_msg;
        }
    } else {
        $_SESSION['error'] = "Employee not found";
    }
    header('Location: gusto_employees.php' . ($page > 1 ? "?page=$page" : "")); 
    exit();
}

// Handle employee deactivation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['deactivate_employee'])) {
    $emp_id = intval($_POST['emp_id']);
    $stmt = mysqli_prepare($con, "UPDATE usermst SET is_active = 0 WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "i", $emp_id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    $_SESSION['success'] = "Employee deactivated successfully.";
    header('Location: gusto_employees.php' . ($page > 1 ? "?page=$page" : "")); 
    exit();
}

// Handle employee activation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['activate_employee'])) {
    $emp_id = intval($_POST['emp_id']);
    $stmt = mysqli_prepare($con, "UPDATE usermst SET is_active = 1 WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "i", $emp_id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    $_SESSION['success'] = "Employee activated successfully.";
    header('Location: gusto_employees.php' . ($page > 1 ? "?page=$page" : "")); 
    exit();
}

// Build SELECT query
$select_query = "SELECT 
    u.id, 
    u.vName, 
    u.vEmail, 
    u.gusto_employee_uuid, 
    u.is_active, 
    u.state, 
    u.home_address_state,
    COALESCE(u.dAmount, 16.00) as hourly_rate,
    COALESCE(u.solo_rate, 3.00) as solo_rate,
    COALESCE(u.miles_rate, 0.25) as miles_rate,
    COALESCE(u.miscellaneous_rate, 1.00) as miscellaneous_rate
FROM usermst u 
$where_clause 
ORDER BY u.vName 
LIMIT $offset, $records_per_page";

// Fetch employees with pagination
$emps_query = mysqli_query($con, $select_query);

// Debug: Check if query failed
if (!$emps_query) {
    die("Employee query failed: " . mysqli_error($con));
}

$employee_count = mysqli_num_rows($emps_query);

// Get custom miles count
$custom_miles_count_result = mysqli_query($con, "SELECT COUNT(*) as count FROM usermst WHERE miles_rate IS NOT NULL AND miles_rate != 0.25");
$custom_miles_count = $custom_miles_count_result ? mysqli_fetch_assoc($custom_miles_count_result)['count'] : 0;

// Get custom miscellaneous count
$custom_misc_count_result = mysqli_query($con, "SELECT COUNT(*) as count FROM usermst WHERE miscellaneous_rate IS NOT NULL AND miscellaneous_rate != 1.00");
$custom_misc_count = $custom_misc_count_result ? mysqli_fetch_assoc($custom_misc_count_result)['count'] : 0;

// DEBUG: Check Amanda Moneymaker's rates specifically
$amanda_check = mysqli_query($con, "SELECT id, vName, dAmount, solo_rate, miles_rate, miscellaneous_rate FROM usermst WHERE vName LIKE '%Amanda%' OR vName LIKE '%Moneymaker%'");
if ($amanda_check && mysqli_num_rows($amanda_check) > 0) {
    $amanda_data = mysqli_fetch_assoc($amanda_check);
    error_log("DEBUG Amanda Rates - ID: {$amanda_data['id']}, Hourly: {$amanda_data['dAmount']}, Solo: {$amanda_data['solo_rate']}, Miles: {$amanda_data['miles_rate']}, Misc: {$amanda_data['miscellaneous_rate']}");
}
?>

<?php include('header.php'); ?>

<!-- Debug Information -->
<div class="debug-info">
    <strong>Debug Info:</strong> 
    Total Records: <?= $total_records ?> | 
    Current Page: <?= $page ?> | 
    Employees Found: <?= $employee_count ?> |
    Custom Miles Rates: <?= $custom_miles_count ?> |
    Custom Misc Rates: <?= $custom_misc_count ?>
</div>

<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h3 class="mb-1">Employee Management</h3>
        <p class="text-muted mb-0">Manage employee Gusto mappings and pay rates</p>
    </div>
    <div class="text-right">
        <div class="btn-group">
            <a href="gusto_weekly_reports_enhanced.php" class="btn btn-outline-primary">
                <i class="fa fa-file-text"></i> Weekly Reports
            </a>
            <a href="gusto_payroll_reports.php" class="btn btn-outline-info">
                <i class="fa fa-bar-chart"></i> Payroll Reports
            </a>
            <a href="rate_audit.php" class="btn btn-outline-warning">
                <i class="fa fa-search"></i> Rate Audit
            </a>
        </div>
    </div>
</div>

<!-- Rate Information Card -->
<div class="card employee-card mb-4">
    <div class="card-header card-header-gradient">
        <h5 class="mb-0">Pay Rate Information</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-3">
                <div class="card bg-light">
                    <div class="card-body text-center">
                        <h6 class="card-title">Standard Rates</h6>
                        <p class="mb-1">Hourly: <strong>$<?= number_format($config->DEFAULT_HOURLY_RATE ?? 16.00, 2) ?></strong></p>
                        <p class="mb-1">Solo: <strong>$<?= number_format($config->DEFAULT_SOLO_RATE ?? 3.00, 2) ?></strong></p>
                        <p class="mb-1">Miles: <strong>$<?= number_format($config->DEFAULT_MILES_RATE ?? 0.25, 2) ?></strong>/mile</p>
                        <p class="mb-0">Misc: <strong>$<?= number_format($config->DEFAULT_MISC_RATE ?? 1.00, 2) ?></strong>/unit</p>
                    </div>
                </div>
            </div>
            <div class="col-md-9">
                <div class="alert alert-info">
                    <h6><i class="fa fa-info-circle"></i> Rate Priority System</h6>
                    <p class="mb-1"><strong>Miles Rate:</strong> Employee Rate → Standard Rate ($0.25/mile)</p>
                    <p class="mb-1"><strong>Solo Rate:</strong> Employee-specific rate (default: $3.00/hr)</p>
                    <p class="mb-0"><strong>Miscellaneous Rate:</strong> Employee-specific rate per unit (default: $1.00/unit)</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Stats Cards -->
<div class="row mb-4">
    <div class="col-xl-3 col-md-6">
        <div class="card stats-card">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h4 class="mb-0"><?= number_format($total_records) ?></h4>
                        <p class="mb-0">Total Employees</p>
                    </div>
                    <div class="flex-shrink-0">
                        <i class="fa fa-users fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="card bg-success text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h4 class="mb-0">
                            <?php 
                            $active_count = mysqli_fetch_assoc(mysqli_query($con, 
                                "SELECT COUNT(*) as count FROM usermst WHERE is_active = 1"
                            ))['count'];
                            echo number_format($active_count);
                            ?>
                        </h4>
                        <p class="mb-0">Active Employees</p>
                    </div>
                    <div class="flex-shrink-0">
                        <i class="fa fa-check-circle fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="card bg-info text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h4 class="mb-0"><?= number_format($custom_miles_count) ?></h4>
                        <p class="mb-0">Custom Miles Rates</p>
                    </div>
                    <div class="flex-shrink-0">
                        <i class="fa fa-car fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="col-xl-3 col-md-6">
        <div class="card bg-warning text-white">
            <div class="card-body">
                <div class="d-flex align-items-center">
                    <div class="flex-grow-1">
                        <h4 class="mb-0"><?= number_format($custom_misc_count) ?></h4>
                        <p class="mb-0">Custom Misc Rates</p>
                    </div>
                    <div class="flex-shrink-0">
                        <i class="fa fa-plus-circle fa-2x opacity-50"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Search and Filter Section -->
<div class="card employee-card mb-4">
    <div class="card-header card-header-gradient">
        <h5 class="mb-0">Search & Filter</h5>
    </div>
    <div class="card-body">
        <form method="GET" class="row g-3 align-items-end">
            <div class="col-md-4">
                <label class="form-label">Search Employees</label>
                <input type="text" name="search" class="form-control search-box" 
                       placeholder="Search by name or email..." value="<?= htmlspecialchars($search) ?>">
            </div>
            <div class="col-md-3">
                <label class="form-label">Status Filter</label>
                <select name="status" class="form-control">
                    <option value="all" <?= $status_filter === 'all' ? 'selected' : '' ?>>All Employees</option>
                    <option value="active" <?= $status_filter === 'active' ? 'selected' : '' ?>>Active Only</option>
                    <option value="inactive" <?= $status_filter === 'inactive' ? 'selected' : '' ?>>Inactive Only</option>
                </select>
            </div>
            <div class="col-md-3">
                <button type="submit" class="btn btn-primary">
                    <i class="fa fa-search"></i> Apply Filters
                </button>
                <?php if ($search || $status_filter !== 'active'): ?>
                    <a href="gusto_employees.php" class="btn btn-outline-secondary">
                        Clear
                    </a>
                <?php endif; ?>
            </div>
        </form>
    </div>
</div>

<!-- Employees Table -->
<div class="card employee-card">
    <div class="card-header card-header-gradient d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Employee List</h5>
        <span class="badge bg-light text-dark">
            Page <?= $page ?> of <?= $total_pages ?> 
            (<?= number_format($total_records) ?> total employees)
        </span>
    </div>
    <div class="card-body">
        <?php if ($employee_count > 0): ?>
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th width="5%">ID</th>
                            <th width="18%">Employee</th>
                            <th width="10%">Hourly Rate</th>
                            <th width="10%">Solo Rate</th>
                            <th width="12%">Miles Rate</th>
                            <th width="10%">Misc Rate</th>
                            <th width="15%">Gusto UUID</th>
                            <th width="20%">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php 
                        while ($e = mysqli_fetch_assoc($emps_query)): 
                            $is_active = $e['is_active'] ?? 1;
                            $is_mapped = !empty($e['gusto_employee_uuid']);
                            $hourly_rate = $e['hourly_rate'] ?? 16.00;
                            $solo_rate = $e['solo_rate'] ?? 3.00;
                            $miles_rate = $e['miles_rate'] ?? 0.25;
                            $miscellaneous_rate = $e['miscellaneous_rate'] ?? 1.00;
                            $has_custom_miles = !empty($miles_rate) && $miles_rate != 0.25;
                            $has_custom_misc = !empty($miscellaneous_rate) && $miscellaneous_rate != 1.00;
                        ?>
                        <tr class="<?= $is_active ? 'active-employee' : 'inactive-employee' ?>">
                            <td>
                                <strong><?= $e['id'] ?></strong>
                            </td>
                            <td class="employee-info">
                                <div class="employee-name"><?= htmlspecialchars($e['vName'] ?? 'No Name') ?></div>
                                <div class="employee-email"><?= htmlspecialchars($e['vEmail'] ?? 'No Email') ?></div>
                            </td>
                            <td class="rate-cell">
                                <form method="post" class="rate-form">
                                    <input type="hidden" name="emp_id" value="<?= $e['id'] ?>">
                                    <div class="input-group input-group-sm mb-1">
                                        <span class="input-group-text">$</span>
                                        <input type="number" name="hourly_rate" step="0.01" min="0" 
                                               value="<?= number_format($hourly_rate, 2) ?>" 
                                               class="form-control rate-input">
                                    </div>
                            </td>
                            <td class="rate-cell">
                                <div class="input-group input-group-sm mb-1">
                                    <span class="input-group-text">$</span>
                                    <input type="number" name="solo_rate" step="0.01" min="0" 
                                           value="<?= number_format($solo_rate, 2) ?>" 
                                           class="form-control rate-input">
                                </div>
                                <?php if ($solo_rate == 3.00): ?>
                                    <small class="default-rate">Default</small>
                                <?php else: ?>
                                    <small class="rate-display">Custom</small>
                                <?php endif; ?>
                            </td>
                            <td class="rate-cell">
                                <div class="input-group input-group-sm mb-1 <?= $has_custom_miles ? 'custom-rate' : '' ?>">
                                    <span class="input-group-text">$</span>
                                    <input type="number" name="custom_miles_rate" step="0.01" min="0" 
                                           value="<?= $has_custom_miles ? number_format($miles_rate, 2) : '' ?>" 
                                           placeholder="0.25" 
                                           class="form-control rate-input">
                                </div>
                                <?php if ($has_custom_miles): ?>
                                    <small class="rate-display">Custom</small>
                                <?php else: ?>
                                    <small class="default-rate">Standard</small>
                                <?php endif; ?>
                            </td>
                            <td class="rate-cell">
                                <div class="input-group input-group-sm mb-1 <?= $has_custom_misc ? 'misc-rate' : '' ?>">
                                    <span class="input-group-text">$</span>
                                    <input type="number" name="miscellaneous_rate" step="0.01" min="0" 
                                           value="<?= number_format($miscellaneous_rate, 2) ?>" 
                                           class="form-control rate-input">
                                </div>
                                <?php if ($has_custom_misc): ?>
                                    <small class="rate-display">Custom</small>
                                <?php else: ?>
                                    <small class="default-rate">Standard</small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if($is_mapped): ?>
                                    <code class="text-success" style="font-size: 0.75rem;">
                                        <?= substr($e['gusto_employee_uuid'], 0, 8) ?>...
                                    </code>
                                    <br>
                                    <span class="badge mapped-badge status-badge">
                                        <i class="fa fa-check"></i> Mapped
                                    </span>
                                <?php else: ?>
                                    <span class="text-muted">Not mapped</span>
                                    <br>
                                    <span class="badge not-mapped-badge status-badge">
                                        <i class="fa fa-times"></i> Not Mapped
                                    </span>
                                <?php endif; ?>
                            </td>
                            <td class="action-cell">
                                <div class="employee-actions mb-2">
                                    <button type="submit" name="update_rates" class="btn btn-primary btn-sm">
                                        <i class="fa fa-save"></i> Update Rates
                                    </button>
                                </div>
                                </form>
                                
                                <div class="employee-actions">
                                <?php if (!$is_mapped): ?>
                                    <!-- Manual UUID Mapping Form -->
                                    <form method="post" class="action-form mb-2">
                                        <input type="hidden" name="emp_id" value="<?= $e['id'] ?>">
                                        <input type="text" name="gusto_uuid" 
                                               placeholder="Paste Gusto UUID" 
                                               class="form-control form-control-sm uuid-input" 
                                               required>
                                        <button type="submit" name="save_map" 
                                                class="btn btn-primary btn-sm"
                                                onclick="return confirm('Save this Gusto UUID mapping?')">
                                            <i class="fa fa-save"></i> Save
                                        </button>
                                    </form>
                                    
                                    <!-- Create in Gusto Form -->
                                    <form method="post" class="action-form">
                                        <input type="hidden" name="emp_id_create" value="<?= $e['id'] ?>">
                                        <button type="submit" name="create_gusto" 
                                                class="btn btn-success btn-sm"
                                                onclick="return confirm('Create this employee in Gusto?')">
                                            <i class="fa fa-plus"></i> Create in Gusto
                                        </button>
                                    </form>
                                <?php else: ?>
                                    <?php if ($is_active): ?>
                                        <form method="post" class="action-form">
                                            <input type="hidden" name="emp_id" value="<?= $e['id'] ?>">
                                            <button type="submit" name="deactivate_employee" 
                                                    class="btn btn-warning btn-sm"
                                                    onclick="return confirm('Deactivate this employee? They will not appear in future payrolls.')">
                                                <i class="fa fa-user-times"></i> Deactivate
                                            </button>
                                        </form>
                                    <?php else: ?>
                                        <form method="post" class="action-form">
                                            <input type="hidden" name="emp_id" value="<?= $e['id'] ?>">
                                            <button type="submit" name="activate_employee" 
                                                    class="btn btn-info btn-sm">
                                                <i class="fa fa-user-plus"></i> Activate
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                <?php endif; ?>
                                </div>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <div class="pagination-container">
                <div class="text-muted">
                    Showing <?= min($records_per_page, $total_records - $offset) ?> of <?= number_format($total_records) ?> employees
                </div>
                <nav>
                    <ul class="pagination pagination-sm">
                        <!-- First Page -->
                        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=1&search=<?= urlencode($search) ?>&status=<?= $status_filter ?>">
                                <i class="fa fa-angle-double-left"></i>
                            </a>
                        </li>
                        
                        <!-- Previous Page -->
                        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= $status_filter ?>">
                                <i class="fa fa-angle-left"></i>
                            </a>
                        </li>

                        <!-- Page Numbers -->
                        <?php
                        $start_page = max(1, $page - 2);
                        $end_page = min($total_pages, $page + 2);
                        
                        for ($i = $start_page; $i <= $end_page; $i++): 
                        ?>
                            <li class="page-item <?= $i == $page ? 'active' : '' ?>">
                                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= $status_filter ?>">
                                    <?= $i ?>
                                </a>
                            </li>
                        <?php endfor; ?>

                        <!-- Next Page -->
                        <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= $status_filter ?>">
                                <i class="fa fa-angle-right"></i>
                            </a>
                        </li>
                        
                        <!-- Last Page -->
                        <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                            <a class="page-link" href="?page=<?= $total_pages ?>&search=<?= urlencode($search) ?>&status=<?= $status_filter ?>">
                                <i class="fa fa-angle-double-right"></i>
                            </a>
                        </li>
                    </ul>
                </nav>
            </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="text-center py-5">
                <i class="fa fa-users fa-4x text-muted mb-3"></i>
                <h4>No Employees Found</h4>
                <p class="text-muted">
                    <?php if (!empty($search) || $status_filter !== 'active'): ?>
                        No employees match your current filters. 
                        <a href="gusto_employees.php" class="btn btn-sm btn-outline-primary ml-2">Clear Filters</a>
                    <?php else: ?>
                        No employees found in the system.
                    <?php endif; ?>
                </p>
            </div>
        <?php endif; ?>
    </div>
</div>

<?php include('footer.php'); ?>

<style>
/* Flash Message Styles - Big and Centered */
.flash-message-container {
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    z-index: 9999;
    width: 90%;
    max-width: 600px;
    animation: flashMessage 0.5s ease-out;
}

.flash-message {
    font-size: 1.5rem;
    font-weight: bold;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 10px 30px rgba(0,0,0,0.3);
    text-align: center;
    border: none;
    margin-bottom: 20px;
    position: relative;
}

.flash-success {
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
    border-left: 10px solid #1e7e34;
}

.flash-error {
    background: linear-gradient(135deg, #dc3545, #e83e8c);
    color: white;
    border-left: 10px solid #bd2130;
}

.flash-warning {
    background: linear-gradient(135deg, #ffc107, #fd7e14);
    color: white;
    border-left: 10px solid #d39e00;
}

.flash-icon {
    font-size: 3rem;
    margin-bottom: 15px;
    display: block;
}

.flash-close {
    background: rgba(255,255,255,0.2);
    border: none;
    color: white;
    font-size: 1.5rem;
    padding: 5px 15px;
    border-radius: 50%;
    position: absolute;
    top: 10px;
    right: 15px;
    cursor: pointer;
    transition: all 0.3s ease;
    z-index: 10000;
}

.flash-close:hover {
    background: rgba(255,255,255,0.3);
    transform: scale(1.1);
}

@keyframes flashMessage {
    0% {
        opacity: 0;
        transform: translate(-50%, -60%) scale(0.8);
    }
    50% {
        opacity: 1;
        transform: translate(-50%, -50%) scale(1.05);
    }
    100% {
        opacity: 1;
        transform: translate(-50%, -50%) scale(1);
    }
}

.flash-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.7);
    z-index: 9998;
    animation: fadeIn 0.3s ease-out;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

/* Existing styles */
.employee-card {
    border: none;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.08);
    margin-bottom: 20px;
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}
.employee-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 20px rgba(0,0,0,0.12);
}
.card-header-gradient {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 10px 10px 0 0 !important;
}
.stats-card {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    color: white;
    border-radius: 10px;
    border: none;
}
.action-form {
    display: flex;
    gap: 8px;
    align-items: center;
    flex-wrap: wrap;
}
.uuid-input {
    min-width: 200px;
    font-family: monospace;
    font-size: 0.85rem;
}
.employee-actions {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
}
.status-badge {
    font-size: 0.75rem;
    padding: 4px 8px;
}
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-top: 20px;
}
.search-box {
    max-width: 300px;
}
.table th {
    background-color: #f8f9fa;
    border-bottom: 2px solid #dee2e6;
    font-weight: 600;
    color: #495057;
    text-align: center;
    vertical-align: middle;
}
.table td {
    vertical-align: middle;
    text-align: center;
}
.mapped-badge {
    background: #28a745;
    color: white;
}
.not-mapped-badge {
    background: #dc3545;
    color: white;
}
.active-employee {
    background-color: #f8fff9 !important;
}
.inactive-employee {
    background-color: #fff8f8 !important;
    opacity: 0.8;
}
.employee-name {
    font-weight: 600;
    color: #2c3e50;
    text-align: left;
}
.employee-email {
    font-size: 0.9rem;
    color: #6c757d;
    text-align: left;
}
.hourly-rate {
    font-weight: 600;
    color: #17a2b8;
}
.rate-input {
    width: 80px;
    text-align: center;
    margin: 0 auto;
}
.rate-form {
    display: flex;
    flex-direction: column;
    gap: 5px;
    align-items: center;
}
.custom-rate {
    background-color: #fff3cd;
    border: 1px solid #ffeaa7;
}
.misc-rate {
    background-color: #e8f5e8;
    border: 1px solid #c3e6cb;
}
.debug-info {
    background: #f8f9fa;
    padding: 10px;
    border-radius: 5px;
    margin-bottom: 15px;
    font-size: 12px;
    border-left: 4px solid #17a2b8;
}
.rate-display {
    font-weight: bold;
    color: #28a745;
    font-size: 0.8rem;
}
.default-rate {
    color: #6c757d;
    font-style: italic;
    font-size: 0.8rem;
}
.form-control.rate-input {
    height: 32px;
    padding: 4px 8px;
}
.input-group-sm {
    height: 32px;
}
.employee-info {
    text-align: left;
}
.rate-cell {
    min-width: 120px;
}
.action-cell {
    min-width: 200px;
}
</style>

<script>
// Flash message functions
function closeFlashMessage() {
    console.log('Closing flash message');
    $('.flash-message-container, .flash-backdrop').fadeOut(300, function() {
        $(this).remove();
    });
}

// Auto-hide flash messages after 5 seconds
$(document).ready(function() {
    // Improved flash message handling
    if ($('.flash-message-container').length > 0) {
        console.log('Flash message found, setting up auto-close');
        
        // Auto-close after 5 seconds
        setTimeout(function() {
            closeFlashMessage();
        }, 5000);

        // Close on close button click
        $(document).on('click', '.flash-close', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Close button clicked');
            closeFlashMessage();
        });

        // Close on backdrop click
        $(document).on('click', '.flash-backdrop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('Backdrop clicked');
            closeFlashMessage();
        });

        // Close on Escape key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape') {
                console.log('Escape key pressed');
                closeFlashMessage();
            }
        });
    }

    // Auto-focus search box
    const searchInput = document.querySelector('input[name="search"]');
    if (searchInput) {
        searchInput.focus();
    }

    // Highlight custom rates on focus
    document.querySelectorAll('.custom-rate input, .misc-rate input').forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.style.backgroundColor = this.parentElement.classList.contains('misc-rate') ? '#e8f5e8' : '#fff3cd';
        });
        input.addEventListener('blur', function() {
            this.parentElement.style.backgroundColor = this.parentElement.classList.contains('misc-rate') ? '#e8f5e8' : '#fff3cd';
        });
    });
});
</script>